/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.mock.interception;

import static org.mule.munit.common.util.Preconditions.checkNotNull;

import static java.util.Arrays.asList;

import static org.mule.runtime.api.component.TypedComponentIdentifier.ComponentType.OPERATION;

import java.util.Collection;
import java.util.Objects;

import org.mule.munit.common.api.model.Event;
import org.mule.munit.common.api.model.EventAttributes;
import org.mule.munit.common.api.model.NullObject;
import org.mule.munit.common.api.model.Payload;
import org.mule.munit.mock.behavior.Behavior;
import org.mule.munit.mock.behavior.CallBehaviour;
import org.mule.munit.mock.behavior.MockBehavior;

import org.mule.runtime.api.component.location.ComponentLocation;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;


/**
 * Validate if a location should be mocked. It also checks if a behaviour definition is valid for a specific location.
 *
 * @author Mulesoft Inc.
 */
class BehaviourValidator {

  private static final String LOGGER = "logger";
  private static final String TRANSFORM = "ee:transform";
  private static final String FLOW_REF = "flow-ref";
  private static final String SET_PAYLOAD = "set-payload";
  private static final String SET_VARIABLE = "set-variable";

  private static final Collection<String> NON_MOCKABLE_CORE_OPERATIONS = asList(LOGGER, TRANSFORM);
  private static final Collection<String> SPECIAL_CORE_OPERATIONS = asList(SET_PAYLOAD, SET_VARIABLE);

  private transient Logger logger = LoggerFactory.getLogger(this.getClass());

  private ComponentLocation location;

  BehaviourValidator(ComponentLocation location) {
    this.location = checkNotNull(location, "The location must not be null.");
  }

  /**
   * Validates if the {@link ComponentLocation} allows to be mocked or not. It works as a blacklist, allowing mock for any
   * component location save those here blacklisted.
   *
   * @return {@code false} if the {@link ComponentLocation} does not allow mocking, {@code true} otherwise
   */
  boolean allowMocking() {
    return OPERATION.equals(location.getComponentIdentifier().getType()) &&
        !NON_MOCKABLE_CORE_OPERATIONS.contains(location.getComponentIdentifier().getIdentifier().toString());
  }

  /**
   * Based on the given {@link ComponentLocation} validate if the {@link Behavior} tries to set an invalid value.
   *
   * @param behaviour a {@link Behavior} to check
   * @return {@code false} if the behaviour attempts to set invalid values for the given location, {@code true} otherwise
   */
  boolean isBehaviorValid(Behavior behaviour) {
    if (!allowMocking()) {
      logger.error(getBaseErrorMessage() + " This component does not allow mocking");

      return false;
    }

    if (behaviour instanceof CallBehaviour && !behaviour.getEvent().isPresent()) {
      logger.error(getBaseErrorMessage() + " CallBehaviour returned no event");

      return false;
    }

    return isBehaviorValidForOperation(behaviour);
  }

  String getBaseErrorMessage() {
    String identifier = location.getComponentIdentifier().getIdentifier().toString();

    return String.format("You have defined a behavior for %s in %s [line: %s].",
                         identifier, location.getFileName().orElse("?"), location.getLineInFile().orElse(-1));
  }

  private boolean isBehaviorValidForOperation(Behavior behaviour) {
    String baseErrorMessage = getBaseErrorMessage() + " This component does not allow definition of";

    if (behaviour instanceof MockBehavior && !behaviour.getEvent().isPresent()) {
      // Should return the same event that was received
      return true;

    } else if (FLOW_REF.equals(location.getComponentIdentifier().getIdentifier().toString())) {
      return true;

    } else if (!isBehaviorValidForCommonNotAllowedElements(behaviour, baseErrorMessage)) {
      return false;

    } else if (SPECIAL_CORE_OPERATIONS.contains(location.getComponentIdentifier().getIdentifier().toString())) {
      return isBehaviorValidForSpecialCoreOperations(behaviour, baseErrorMessage);

    } else {
      return true;
    }
  }

  private boolean isBehaviorValidForSpecialCoreOperations(Behavior behaviour, String baseErrorMessage) {
    String identifier = location.getComponentIdentifier().getIdentifier().toString();

    if (SET_PAYLOAD.equals(identifier)) {
      return isBehaviorValidForSetPayload(behaviour, baseErrorMessage);

    } else {
      return isBehaviorValidForSetVariable(behaviour, baseErrorMessage);
    }
  }

  private boolean isBehaviorValidForSetPayload(Behavior behaviour, String baseErrorMessage) {
    Event event = behaviour.getEvent().orElseThrow(IllegalArgumentException::new);

    if (behaviour instanceof CallBehaviour) {
      Event input = ((CallBehaviour) behaviour).getInput().orElseThrow(IllegalArgumentException::new);

      if (!equals(input.getAttributes(), event.getAttributes())) {
        logger.error(baseErrorMessage + " attributes.");

        return false;

      } else if (!equals(input.getVariables(), event.getVariables())) {
        logger.error(baseErrorMessage + " variables.");

        return false;
      }

    } else {
      if (!equals(event.getAttributes(), null)) {
        logger.error(baseErrorMessage + " attributes.");

        return false;

      } else if (!equals(event.getVariables(), null)) {
        logger.error(baseErrorMessage + " variables.");

        return false;
      }
    }

    return true;
  }

  private boolean isBehaviorValidForSetVariable(Behavior behaviour, String baseErrorMessage) {
    Event event = behaviour.getEvent().orElseThrow(IllegalArgumentException::new);

    if (behaviour instanceof CallBehaviour) {
      Event input = ((CallBehaviour) behaviour).getInput().orElseThrow(IllegalArgumentException::new);

      if (!equals(input.getAttributes(), event.getAttributes())) {
        logger.error(baseErrorMessage + " attributes.");

        return false;

      } else if (!equals(input.getPayload(), event.getPayload())) {
        logger.error(baseErrorMessage + " payload.");

        return false;
      }

    } else {
      if (!equals(event.getAttributes(), null)) {
        logger.error(baseErrorMessage + " attributes.");

        return false;

      } else if (!equals(event.getPayload(), null)) {
        logger.error(baseErrorMessage + " payload.");

        return false;
      }
    }

    return true;
  }

  private boolean isBehaviorValidForCommonNotAllowedElements(Behavior behaviour, String baseErrorMessage) {
    Event event = behaviour.getEvent().orElseThrow(IllegalArgumentException::new);

    if (!equals(event.getSessionProperties(), null)) {
      logger.error(baseErrorMessage + " session properties.");

      return false;

    } else if (!equals(event.getOutboundProperties(), null)) {
      logger.error(baseErrorMessage + " outbound properties.");

      return false;

    } else if (!equals(event.getInboundProperties(), null)) {
      logger.error(baseErrorMessage + " inbound properties.");

      return false;

    } else if (!equals(event.getInboundAttachments(), null)) {
      logger.error(baseErrorMessage + " inbound attachments.");

      return false;

    } else if (!equals(event.getOutboundAttachments(), null)) {
      logger.error(baseErrorMessage + " outbound attachments.");

      return false;
    }

    return true;
  }

  @SuppressWarnings("BooleanMethodIsAlwaysInverted")
  private boolean equals(Payload first, Payload second) {
    return Objects.equals(first, second)
        || (first == null && second.getValue() instanceof NullObject)
        || (second == null && first.getValue() instanceof NullObject);
  }

  @SuppressWarnings("BooleanMethodIsAlwaysInverted")
  private boolean equals(EventAttributes first, EventAttributes second) {
    return Objects.equals(first, second)
        || (first == null && second.getValue() instanceof NullObject)
        || (second == null && first.getValue() instanceof NullObject);
  }

  @SuppressWarnings("BooleanMethodIsAlwaysInverted")
  private <T> boolean equals(Collection<T> first, Collection<T> second) {
    return Objects.equals(first, second)
        || (first == null && second.isEmpty())
        || (second == null && first.isEmpty());
  }
}
