/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.mock.interception;

import static org.mule.runtime.api.component.TypedComponentIdentifier.ComponentType.OPERATION;

import java.util.Arrays;
import java.util.List;

import javax.inject.Inject;

import org.mule.munit.common.behavior.BehaviorManager;
import org.mule.runtime.api.component.ComponentIdentifier;
import org.mule.runtime.api.component.TypedComponentIdentifier;
import org.mule.runtime.api.component.location.ComponentLocation;
import org.mule.runtime.api.exception.ErrorTypeRepository;
import org.mule.runtime.api.interception.ProcessorInterceptor;
import org.mule.runtime.api.interception.ProcessorInterceptorFactory;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * The goal of this class is to define if a processor should be intercepted. If so creates a ProcessorInterceptor
 *
 * @author Mulesoft Inc.
 * @since 2.0.0
 */
public class MunitProcessorInterceptorFactory implements ProcessorInterceptorFactory {

  private static final String MUNIT_NAMESPACE = "munit";
  private static final String MUNIT_TOOLS_NAMESPACE = "munit-tools";
  private static final List<String> NON_INTERCEPTABLE_NAMESPACES = Arrays.asList(MUNIT_NAMESPACE, MUNIT_TOOLS_NAMESPACE);

  private transient Logger logger = LoggerFactory.getLogger(this.getClass());

  private BehaviorManager behaviorManager;
  private ErrorTypeRepository errorTypeRepository;

  @Inject
  @SuppressWarnings("unused")
  public void setBehaviorManager(BehaviorManager behaviorManager) {
    this.behaviorManager = behaviorManager;
  }

  @Inject
  @SuppressWarnings("unused")
  public void setErrorTypeRepository(ErrorTypeRepository errorTypeRepository) {
    this.errorTypeRepository = errorTypeRepository;
  }

  @Override
  public ProcessorInterceptor get() {
    return createInterceptor(behaviorManager, errorTypeRepository);
  }

  @Override
  public boolean intercept(ComponentLocation location) {
    TypedComponentIdentifier identifier = location.getComponentIdentifier();
    String name = identifier.getIdentifier().getName();
    if (NON_INTERCEPTABLE_NAMESPACES.contains(identifier.getIdentifier().getNamespace())) {
      logger.debug("Processor {} not intercepted. Non interceptable namespace.", name);
      return false;
    }
    if (!OPERATION.equals(identifier.getType())) {
      logger.debug("Processor {} not intercepted. Component is not an operation.", name);
      return false;
    }
    if (isInsideMunitToolsSpy(location)) {
      logger.debug("Processor {} not intercepted. Component is located inside an MUnit Spy processor.", name);
      return false;
    }
    logger.debug("Processor {} successfully intercepted.", name);
    return true;
  }

  protected MunitProcessorInterceptor createInterceptor(BehaviorManager behaviorManager,
                                                        ErrorTypeRepository errorTypeRepository) {

    MunitProcessorInterceptor interceptor = new MunitProcessorInterceptor();
    interceptor.setManager(behaviorManager);
    interceptor.setErrorTypeRepository(errorTypeRepository);
    return interceptor;
  }

  private boolean isInsideMunitToolsSpy(ComponentLocation location) {
    return location.getParts().stream()
        .anyMatch(locationPart -> locationPart.getPartIdentifier().map(this::isMunitToolsSpy).orElse(false));
  }

  private boolean isMunitToolsSpy(TypedComponentIdentifier identifier) {
    ComponentIdentifier componentIdentifier = identifier.getIdentifier();
    return MUNIT_TOOLS_NAMESPACE.equals(componentIdentifier.getNamespace()) && "spy".equals(componentIdentifier.getName());
  }
}
