/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.mock.tool;

import org.mule.munit.common.behavior.ProcessorCall;
import org.mule.munit.common.behavior.ProcessorId;
import org.mule.munit.common.api.model.Event;
import org.mule.munit.common.api.model.FlowName;
import org.mule.munit.mock.behavior.CallBehaviour;
import org.mule.munit.mock.behavior.DefaultBehaviorManager;
import org.mule.munit.mock.behavior.MockBehavior;
import org.mule.runtime.api.artifact.Registry;
import org.mule.runtime.core.api.construct.Flow;

import java.util.Map;
import java.util.Optional;

/**
 * Mocking Tool to create processor mocks.
 *
 * Usage: new ProcessorMocker(muleContext).when("processors").ofNamespace("namespace").thenReturn(muleMessage);
 *
 * @author Mulesoft Inc.
 * @since 1.0.0
 */
public class ProcessorMocker extends AbstractMockingTool {

  public ProcessorMocker(Registry registry) {
    super(registry);
  }

  /**
   * Defines the name of the processor to be mocked
   *
   * @param name The name of the processor to be mocked
   * @return {@code this}
   */
  public ProcessorMocker when(String name) {
    this.processorName = name;
    return this;
  }

  /**
   * Defines the namespace of the processor to be mocked
   *
   * @param namespace The namespace of the processor to be mocked
   *
   * @return {@code this}
   */
  public ProcessorMocker ofNamespace(String namespace) {
    this.processorNamespace = namespace;
    return this;
  }

  /**
   * Defines the attributes of the processor to be mocked
   *
   * @param attributes The attributes of the processor to be mocked
   * @return {@code this}
   */
  public ProcessorMocker withAttributes(Map<String, Object> attributes) {
    this.processorAttributes = attributes;
    return this;
  }

  /**
   * Defines what {@link Event} to return after the processor call
   *
   * @param event The Event to return
   */
  // TODO change this to return events MU-641
  public void thenReturn(Event event) {
    validateMessageProcessorName();

    ProcessorCall processorCall = new ProcessorCall(new ProcessorId(processorName, processorNamespace));
    processorCall.setAttributes(processorAttributes);
    getManager().addBehavior(new MockBehavior(processorCall, event));
  }

  public void thenCall(FlowName flowName) {
    validateMessageProcessorName();

    ProcessorCall processorCall = new ProcessorCall(new ProcessorId(processorName, processorNamespace));
    processorCall.setAttributes(processorAttributes);
    getManager().addBehavior(new CallBehaviour(processorCall, getFlow(flowName.getFlow())));
  }

  /**
   * Determines that the mocked processor must return the same event as before its call.
   */
  public void thenReturnSameEvent() {
    validateMessageProcessorName();

    ProcessorCall processorCall = new ProcessorCall(new ProcessorId(processorName, processorNamespace));
    processorCall.setAttributes(processorAttributes);

    DefaultBehaviorManager manager = getManager();
    manager.addBehavior(new MockBehavior(processorCall));
  }

  private void validateMessageProcessorName() {
    if (processorName == null) {
      throw new IllegalArgumentException("You must specify at least the processor name");
    }
  }

  private Flow getFlow(String flowName) {
    Optional<Object> object = registry.lookupByName(flowName);

    if (object.isPresent() && object.get() instanceof Flow) {
      return (Flow) object.get();
    }

    throw new IllegalArgumentException(String.format("Invalid flow name: '%s'. HINT: only flows can be used, not sub-flows",
                                                     flowName));
  }
}
