/*
 * Copyright (c) 2015 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.remote;


import org.apache.commons.cli.ParseException;
import org.apache.commons.io.FilenameUtils;
import org.json.JSONObject;
import org.mule.munit.plugins.coverage.CoberturaPlugin;
import org.mule.munit.plugins.coverage.report.ApplicationCoverageReport;
import org.mule.munit.remote.config.RunConfiguration;
import org.mule.munit.remote.config.RunConfigurationBuilder;
import org.mule.munit.remote.listener.MessageBuilderEventListener;
import org.mule.munit.remote.listener.RemoteRunnerEventListener;
import org.mule.munit.remote.notification.ConsoleMessageHandler;
import org.mule.munit.remote.notification.MessageHandlerContainer;
import org.mule.munit.remote.notification.SocketMessageHandler;
import org.mule.munit.runner.mule.SuiteRunner;
import org.mule.munit.runner.mule.result.notification.StreamSuiteRunnerEventListener;
import org.mule.munit.runner.mule.result.notification.SuiteRunnerEventListener;
import org.mule.munit.runner.mule.result.notification.SuiteRunnerEventListenerContainer;

import java.io.IOException;
import java.util.List;

import static org.mule.munit.remote.notification.MessageHandlerMode.CONSOLE;
import static org.mule.munit.remote.notification.MessageHandlerMode.SOCKET;

//TODO put a freaking doc
public class RemoteRunner {

    public static void main(String args[]) throws ParseException, IOException {
        RunConfiguration runConfig = new RunConfigurationBuilder(args).build();
        RemoteRunner runner = new RemoteRunner();
        runner.run(runConfig);
        shutDown();
    }

    public void run(RunConfiguration runConfig) throws IOException {
        System.out.println("[" + this.getClass().getName() + "]" + "Run Started");

        CoverageManager coverageManager = buildCoverageManager(runConfig);
        coverageManager.startCoverageServer();

        MessageBuilderEventListener listener = buildRunnerListener(runConfig);
        listener.notifyRunStart();

        List<String> testNameList = runConfig.getTestList();
        for (String munitSuite : runConfig.getSuiteList()) {
            try {
                runTestSuite(munitSuite, testNameList, runConfig.getProjectName(), listener);
            } catch (RuntimeException e) {
                e.printStackTrace();
            }
        }

        listener.notifyRunFinish();

        coverageManager.stopCoverageServer();

        calculateCoverageIfApplicable(coverageManager, listener);
        System.out.println("[" + this.getClass().getName() + "]" + "Done");
    }

    private void runTestSuite(String munitSuite, List<String> testNames, String projectName, MessageBuilderEventListener listener) {
        String suitePath = FilenameUtils.getFullPath(munitSuite);
        String suiteName = FilenameUtils.getName(munitSuite);

        listener.defineCurrentSuite(suitePath, suiteName);

        SuiteRunner runner = new SuiteRunner(munitSuite, testNames, projectName, buildListenerContainer(listener));
        runner.run();
    }

    private CoverageManager buildCoverageManager(RunConfiguration runConfig) {
        boolean shouldCalculateCoverage = runConfig.isRunCoverage();
        String projectName = runConfig.getProjectName();
        CoverageManager coverageManager;
        if (!shouldCalculateCoverage) {
            coverageManager = new CoverageManager(-1, "", false, projectName);
        } else {
            int coveragePort = runConfig.getCoveragePort();
            System.setProperty(CoberturaPlugin.COBERTURA_PORT_PROPERTY, String.valueOf(coveragePort));
            String applicationResources = runConfig.getApplicationPaths();

            System.out.format("Coverage port: %d resources: %s \n", coveragePort, applicationResources);
            coverageManager = new CoverageManager(coveragePort, applicationResources, true, projectName);
            coverageManager.setIgnoreFlows(runConfig.getIgnoreFlows());
        }
        return coverageManager;
    }

    private SuiteRunnerEventListener buildListenerContainer(SuiteRunnerEventListener listener) {
        SuiteRunnerEventListenerContainer listenerDecorator = new SuiteRunnerEventListenerContainer();
        listenerDecorator.addNotificationListener(listener);
        listenerDecorator.addNotificationListener(new StreamSuiteRunnerEventListener(System.out));
        return listenerDecorator;
    }

    private MessageBuilderEventListener buildRunnerListener(RunConfiguration runConfig) throws IOException {
        MessageHandlerContainer messageHandlerContainer = new MessageHandlerContainer();
        for (String notifier : runConfig.getNotifiers()) {
            if (CONSOLE.toString().equals(notifier.toLowerCase())) {
                messageHandlerContainer.addMessageHandler(new ConsoleMessageHandler());
            } else if (SOCKET.toString().equals(notifier.toLowerCase())) {
                messageHandlerContainer.addMessageHandler(new SocketMessageHandler(runConfig.getPort()));
            } else {
                System.out.format("Mode '%s' does not exist\n", notifier);
            }
        }
        return new MessageBuilderEventListener(runConfig.getRunToken(), messageHandlerContainer);
    }

    private void calculateCoverageIfApplicable(CoverageManager coverageManager, RemoteRunnerEventListener listener) {
        ApplicationCoverageReport coverageReport = coverageManager.generateCoverageReport();
        String coverageReportJson = new JSONObject(coverageReport).toString();
        listener.notifyCoverageReport(coverageReportJson);
    }


    private static void shutDown() {
        System.exit(0);
    }
}
