package org.mule.munit.remote.listener;

import org.apache.commons.lang.Validate;
import org.mule.munit.remote.MessageBuilder;
import org.mule.munit.remote.RemoteRunnerMessage;
import org.mule.munit.remote.notification.MessageHandler;
import org.mule.munit.runner.mule.MunitTest;
import org.mule.munit.runner.mule.result.SuiteResult;
import org.mule.munit.runner.mule.result.TestResult;
import org.mule.munit.runner.mule.result.notification.Notification;
import org.mule.munit.runner.mule.result.notification.SuiteRunnerEventListener;

public class MessageBuilderEventListener implements RemoteRunnerEventListener, SuiteRunnerEventListener {

    private MessageHandler notifier;

    private String runToken;

    private String suitePath;
    private String suiteName;

    public MessageBuilderEventListener(String runToken, MessageHandler notifier) {
        Validate.notNull(notifier, "The notifier can not be null");
        Validate.notEmpty(runToken, "The run token can not be null nor empty");

        this.notifier = notifier;
        this.runToken = runToken;
    }

    public synchronized void defineCurrentSuite(String suitePath, String suiteName) {
        Validate.notEmpty(suitePath, "The suite path can not be null nor empty");
        Validate.notEmpty(suiteName, "The suite name can not be null nor empty");

        this.suitePath = suitePath;
        this.suiteName = suiteName;
    }

    public void notifyRunStart() {
        String message = MessageBuilder.runStartMessage(runToken);
        sendMessage(message);
    }

    public void notifySuiteStart(String suite) {
        String message = MessageBuilder.testSuiteStartMessage(runToken, suitePath, suiteName);
        sendMessage(message);
    }

    public void notifySuiteStartFailure(Notification notification) {
        String message = MessageBuilder.testSuiteStartFailureMessage(runToken, suitePath, suiteName, notification.getFullMessage());
        sendMessage(message);
    }

    public void notifyNumberOfTests(int numberOfTests) {
        String message = MessageBuilder.numberOfTestsMessage(runToken, suitePath, suiteName, String.valueOf(numberOfTests));
        sendMessage(message);
    }

    public void notifyTestStart(MunitTest test) {
        String message = MessageBuilder.testStartMessage(runToken, suitePath, suiteName, test.getName());
        sendMessage(message);
    }

    public void notifyTestResult(TestResult testResult) {
        String message;

        if (testResult.getError() != null) {
            message = MessageBuilder.testErrorMessage(runToken, suitePath, suiteName, testResult.getTestName(), testResult.getError().getFullMessage());
        } else if (testResult.getFailure() != null) {
            message = MessageBuilder.testFailureMessage(runToken, suitePath, suiteName, testResult.getTestName(), testResult.getFailure().getFullMessage());
        } else {
            message = MessageBuilder.testFinishedMessage(runToken, suitePath, suiteName, testResult.getTestName());
        }

        sendMessage(message);
    }

    public void notifyTestIgnored(TestResult testResult) {
        String message = MessageBuilder.testIgnoredMessage(runToken, suitePath, suiteName, testResult.getName());
        sendMessage(message);
    }

    public void notifySuiteEnd(SuiteResult result) {
        String message = MessageBuilder.testSuiteFinishedMessage(runToken, suitePath, suiteName);
        sendMessage(message);
    }

    public void notifyBeforeSuiteFailure(Notification notification) {
        String message = MessageBuilder.beforeSuiteFailureMessage(runToken, suitePath, suiteName, notification.getFullMessage());
        sendMessage(message);
    }

    public void notifyBeforeSuiteError(Notification notification) {
        String message = MessageBuilder.beforeSuiteErrorMessage(runToken, suitePath, suiteName, notification.getFullMessage());
        sendMessage(message);
    }

    public void notifyAfterSuiteFailure(Notification notification) {
        String message = MessageBuilder.afterSuiteFailureMessage(runToken, suitePath, suiteName, notification.getFullMessage());
        sendMessage(message);
    }

    public void notifyAfterSuiteError(Notification notification) {
        String message = MessageBuilder.afterSuiteErrorMessage(runToken, suitePath, suiteName, notification.getFullMessage());
        sendMessage(message);
    }

    public void notifyCoverageReport(String coverageReportJson) {
        String message = MessageBuilder.notifyCoverageReportMessage(runToken, coverageReportJson);
        sendMessage(message);
    }

    public void notifyRunFinish() {
        String message = MessageBuilder.runFinishedMessage(runToken);
        sendMessage(message);
    }


    private void sendMessage(String message) {
        notifier.handle(message);
    }
}
