/*
 * Copyright (c) 2015 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.remote;


import org.apache.commons.io.FilenameUtils;
import org.apache.commons.lang.StringUtils;
import org.json.JSONObject;
import org.mule.munit.plugins.coverage.CoberturaPlugin;
import org.mule.munit.plugins.coverage.report.ApplicationCoverageReport;
import org.mule.munit.plugins.coverage.server.MunitCoverageServer;
import org.mule.munit.runner.mule.MunitSuiteRunner;
import org.mule.munit.runner.mule.MunitTest;
import org.mule.munit.runner.mule.result.notification.Notification;
import org.mule.munit.runner.mule.result.notification.NotificationListener;
import org.mule.munit.runner.mule.result.notification.NotificationListenerDecorator;
import org.mule.munit.runner.mule.result.notification.StreamNotificationListener;

import java.io.IOException;
import java.io.ObjectOutputStream;
import java.net.Socket;
import java.util.*;
@Deprecated
public class MunitRemoteRunner {
    public static final String PORT_PARAMETER = "-port";
    public static final String RESOURCE_PARAMETER = "-resource";
    public static final String RUN_TOKEN_PARAMETER = "-run_token";
    public static final String TEST_NAME_PARAMETER = "-test_name";

    public static final String RESOURCES_TOKEN_SEPARATOR = ",";

    public static final String TEST_NAME_TOKEN_SEPARATOR = "<";

    public static final String MUNIT_PROJECT_NAME_SYSTEM_PROPERTY = "munitProjectName";
    public static final String CALCULATE_COVERAGE_SYSTEM_PROPERTY = "calculate.coverage";
    public static final String APPLICATION_RESOURCES_SYSTEM_PROPERTY = "application.paths";

    protected String message;
    protected Socket requestSocket;
    protected ObjectOutputStream out;

    private Integer coveragePort;
    private MunitCoverageServer coverageServer;
    private Set<String> coveredPaths = new HashSet<String>();


    public static void main(String args[]) {
        int port = -1;

        String runToken = null;
        String resources = null;
        String testName = null;

        for (int i = 0; i < args.length; i++) {
            if (args[i].equalsIgnoreCase(RUN_TOKEN_PARAMETER)) {
                runToken = args[i + 1];
            }

            if (args[i].equalsIgnoreCase(RESOURCE_PARAMETER)) {
                resources = args[i + 1];
            }
            if (args[i].equalsIgnoreCase(PORT_PARAMETER)) {
                port = Integer.valueOf(args[i + 1]);
            }

            if (args[i].equalsIgnoreCase(TEST_NAME_PARAMETER)) {
                testName = args[i + 1];
            }
        }

        MunitRemoteRunner serverRemoteRunner = new MunitRemoteRunner();
        serverRemoteRunner.run(port, runToken, resources, testName);

    }

    /**
     * This method's goal is to run a single test suite file and notify events to a listening server.
     *
     * @param port      the port to connect to notify run status
     * @param runToken  the token to id this specific run
     * @param resources a comma separated list of resource files path to be loaded
     * @param testName  the test name, if any, in the event you wish to run a single test. It's a regex.
     */
    public void run(int port, String runToken, String resources, String testName) {
        RemoteRunnerNotificationListener listener = null;
        CoverageManager coverageManager = buildCoverageManager();
        try {
            connectToStudioServer(port);

            coverageManager.startCoverageServer();

            listener = new RemoteRunnerNotificationListener(runToken, out);

            listener.notifyRunStart();
            List<String> testNameList = buildTestNamesList(testName);
            for (String resource : resources.split(RESOURCES_TOKEN_SEPARATOR)) {
                try {
                    runTestSuite(resource, testNameList, listener);
                } catch (RuntimeException e) {
                    e.printStackTrace();
                }
            }
            coverageManager.stopCoverageServer();
        } catch (IOException ioException) {
            // this only can happen if connection can't be established
            ioException.printStackTrace();
        } finally {
            if (null != listener) {
                calculateCoverageIfApplicable(coverageManager, listener);
                listener.notifyRunFinish();
            }

            closeConnectionToStudioServer();

            System.out.println("[" + this.getClass().getName() + "]" + "Done");
        }
        shutDown();
    }

    protected void shutDown() {
        System.exit(0);
    }

    private List<String> buildTestNamesList(String testNames) {
        List<String> testNameList;
        if (StringUtils.isNotBlank(testNames)) {
            testNameList = Arrays.asList(testNames.split(TEST_NAME_TOKEN_SEPARATOR));
        } else {
            testNameList = new ArrayList();
        }
        return testNameList;
    }

    private void connectToStudioServer(int port) throws IOException {
        requestSocket = new Socket("localhost", port);
        System.out.println("[" + this.getClass().getName() + "]" + "Connected to localhost in port " + port);
        out = new ObjectOutputStream(requestSocket.getOutputStream());
        out.flush();
    }

    private void closeConnectionToStudioServer() {
        try {
            if (null != out) {
                out.close();
            }
            if (null != requestSocket) {
                requestSocket.close();
            }
        } catch (IOException ioException) {
            ioException.printStackTrace();
        }
    }

    private CoverageManager buildCoverageManager() {
        Boolean shouldCalculateCoverage = Boolean.valueOf(System.getProperty(CALCULATE_COVERAGE_SYSTEM_PROPERTY));
        if (!shouldCalculateCoverage) {
            return new CoverageManager(-1, "", false, getProjectName());
        } else {
            String coberturaPort = System.getProperty(CoberturaPlugin.COBERTURA_PORT_PROPERTY);
            String applicationResources = System.getProperty(APPLICATION_RESOURCES_SYSTEM_PROPERTY);

            System.out.println("Cobertura port: " + coberturaPort + " resources: " + applicationResources);

            return new CoverageManager(Integer.valueOf(coberturaPort), applicationResources, true, getProjectName());
        }
    }

    private void calculateCoverageIfApplicable(CoverageManager coverageManager, RemoteRunnerNotificationListener listener) {
        ApplicationCoverageReport coverageReport = coverageManager.generateCoverageReport();
        if (null != listener) {
            String coverageReportJson = new JSONObject(coverageReport).toString();
            listener.notifyCoverageReport(coverageReportJson);
        }
    }

    private int runTestSuite(String resource, List<String> testNames, RemoteRunnerNotificationListener listener) {
        String suitePath = FilenameUtils.getPath(resource);
        String suiteName = FilenameUtils.getName(resource);

        listener.defineCurrentSuite(suitePath, suiteName);
        MunitSuiteRunner runner;
        try {
            runner = new MunitSuiteRunner(resource, testNames, getProjectName());
            runner.setNotificationListener(buildListenerDecorator(listener));

            listener.notifySuiteStart();
            listener.notifyNumberOfTest(runner.getNumberOfTests());
        } catch (RuntimeException e) {
            listener.notifySuiteStartFailure(new Notification(e.getMessage(), MunitTest.stack2string(e)));
            throw e;
        }
        try {
            runner.run();
        } finally {
            listener.notifySuiteFinished();
            return runner.getNumberOfTests();
        }
    }

    private NotificationListenerDecorator buildListenerDecorator(NotificationListener listener) {
        NotificationListenerDecorator listenerDecorator = new NotificationListenerDecorator();
        listenerDecorator.addNotificationListener(listener);
        listenerDecorator.addNotificationListener(new StreamNotificationListener(System.out));
        return listenerDecorator;
    }

    private String getProjectName() {
        return System.getProperty(MUNIT_PROJECT_NAME_SYSTEM_PROPERTY);
    }
}