package org.mule.munit.remote.config;

import org.apache.commons.cli.*;
import org.apache.commons.lang.StringUtils;
import org.mule.munit.common.util.FreePortFinder;

import java.util.*;

/**
 * Takes a Command Line Arguments and generates a runConfiguration
 */
public class RunConfigurationBuilder {


    public static final String SUITES_PARAMETER = "suites";
    public static final String RUN_TOKEN_PARAMETER = "run_token";
    public static final String TEST_NAME_PARAMETER = "test_name";
    public static final String NOTIFIERS_PARAMETER = "mode";
    public static final String PORT_PARAMETER = "port";

    public static final String RUN_COVERAGE_PARAMETER = "run_coverage";
    public static final String COVERAGE_PORT_PARAMETER = "coverage_port";
    public static final String PROJECT_NAME_PARAMETER = "project_name";
    public static final String APPLICATION_PATHS_PARAMETER = "application_paths";
    public static final String IGNORE_FLOWS_PARAMETER = "ignore_flows";

    public static final String SUITES_TOKEN_SEPARATOR = ",";
    public static final String NOTIFIERS_TOKEN_SEPARATOR = ",";
    public static final String IGNORE_FLOWS_TOKEN_SEPARATOR = ",";
    public static final String TESTS_TOKEN_SEPARATOR = "<";

    private static final int MIN_PORT_NUMBER = 50000;
    private static final int MAX_PORT_NUMBER = 55000;



    private String[] args = new String[0];
    private Options options = new Options();
    private CommandLineParser parser = new BasicParser();

    public RunConfigurationBuilder(String[] args) {
        configureOptions();
        this.args = args;
    }

    private void configureOptions() {
        options.addOption(createOption(SUITES_PARAMETER, true, "MUnit suites to be run", true));
        options.addOption(createOption(RUN_TOKEN_PARAMETER, true, "Run Token", true));
        options.addOption(createOption(PORT_PARAMETER, "Port to handle suite results"));
        options.addOption(createOption(NOTIFIERS_PARAMETER, "Mode to handle suite results"));
        options.addOption(createOption(TEST_NAME_PARAMETER, "Test to be run"));
        options.addOption(createOption(RUN_COVERAGE_PARAMETER, false, "Whether is has to run coverage", false));
        options.addOption(createOption(PROJECT_NAME_PARAMETER, "Name of the project"));
        options.addOption(createOption(APPLICATION_PATHS_PARAMETER, "Application files to be covered"));
        options.addOption(createOption(COVERAGE_PORT_PARAMETER, "Port for coverage server"));
        options.addOption(createOption(IGNORE_FLOWS_PARAMETER, "Flows to ignore when running coverage"));
    }

    public RunConfiguration build() throws ParseException {
        CommandLine cmdLine = parser.parse(options, args);
        RunConfiguration runConfig = new RunConfiguration();

        runConfig.setSuiteList(buildSuiteList(cmdLine));
        runConfig.setTestList(buildTestList(cmdLine));
        runConfig.setNotifiers(buildNotifiersList(cmdLine));
        runConfig.setRunToken(cmdLine.getOptionValue(RUN_TOKEN_PARAMETER));
        runConfig.setRunCoverage(cmdLine.hasOption(RUN_COVERAGE_PARAMETER));
        runConfig.setProjectName(cmdLine.getOptionValue(PROJECT_NAME_PARAMETER));
        if (cmdLine.hasOption(PORT_PARAMETER)) {
            runConfig.setPort(Integer.parseInt(cmdLine.getOptionValue(PORT_PARAMETER)));
        }
        if (runConfig.isRunCoverage()) {
            buildCoverageConfiguration(cmdLine, runConfig);
        }

        return runConfig;
    }

    private Option createOption(String name, String description) {
        return createOption(name, true, description, false);
    }

    private Option createOption(String name, boolean hasArg, String description, boolean required) {
        OptionBuilder.withDescription(description);
        OptionBuilder.hasArg(hasArg);
        OptionBuilder.isRequired(required);
        return OptionBuilder.create(name);
    }

    private List<String> buildSuiteList(CommandLine cmd) {
        String suitesParam = cmd.getOptionValue(SUITES_PARAMETER, StringUtils.EMPTY);
        return Arrays.asList(StringUtils.split(suitesParam, SUITES_TOKEN_SEPARATOR));
    }

    private List<String> buildNotifiersList(CommandLine cmd) {
        String suitesParam = cmd.getOptionValue(NOTIFIERS_PARAMETER, StringUtils.EMPTY);
        return Arrays.asList(StringUtils.split(suitesParam, NOTIFIERS_TOKEN_SEPARATOR));
    }

    private Set<String> buildIgnoreFlowsList(CommandLine cmd) {
        String flowsParam = cmd.getOptionValue(IGNORE_FLOWS_PARAMETER, StringUtils.EMPTY);
        return new HashSet<String>(Arrays.asList(StringUtils.split(flowsParam, IGNORE_FLOWS_TOKEN_SEPARATOR)));
    }

    private List<String> buildTestList(CommandLine cmd) {
        String testNames = cmd.getOptionValue(TEST_NAME_PARAMETER, StringUtils.EMPTY);
        if (StringUtils.isEmpty(testNames)) {
            return new ArrayList<String>();
        }
        return Arrays.asList(testNames.split(TESTS_TOKEN_SEPARATOR));
    }

    private void buildCoverageConfiguration(CommandLine cmdLine, RunConfiguration runConfig) {
        runConfig.setApplicationPaths(cmdLine.getOptionValue(APPLICATION_PATHS_PARAMETER));
        Integer coveragePort;
        if (cmdLine.hasOption(COVERAGE_PORT_PARAMETER)) {
            coveragePort = Integer.parseInt(cmdLine.getOptionValue(COVERAGE_PORT_PARAMETER));
        } else {
            FreePortFinder portFinder = new FreePortFinder(MIN_PORT_NUMBER, MAX_PORT_NUMBER);
            coveragePort = portFinder.find();
        }
        runConfig.setCoveragePort(coveragePort);
        runConfig.setIgnoreFlows(buildIgnoreFlowsList(cmdLine));
    }

}
