/*
 * Copyright (c) 2015 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.remote;

import static com.google.common.base.Preconditions.checkArgument;
import static org.apache.commons.io.filefilter.TrueFileFilter.INSTANCE;
import static org.mule.munit.remote.FolderNames.APPLICATION;
import static org.mule.munit.remote.FolderNames.CLASSES;
import static org.mule.munit.remote.FolderNames.META_INF;
import static org.mule.munit.remote.FolderNames.MULE;
import static org.mule.munit.remote.FolderNames.REPOSITORY;
import static org.mule.munit.remote.FolderNames.TEST_CLASSES;
import static org.mule.munit.remote.FolderNames.TEST_MULE;

import java.io.File;
import java.io.IOException;
import java.nio.file.Path;
import java.nio.file.Paths;

import org.apache.commons.io.FileUtils;

/**
 * It create a structure similar to that of an exploded application based on a source folder containing all the required folders
 * 
 * @author Mulesoft Inc.
 * @since 2.0.0
 */
public class ApplicationStructureGenerator {

  private File sourceFolder;
  private File destinationFolder;

  public ApplicationStructureGenerator(File sourceFolder, File destinationFolder) {
    checkArgument(sourceFolder.exists(), "The source folder " + sourceFolder.getAbsolutePath() + " should exists");
    checkArgument(destinationFolder.exists(), "The destination folder " + destinationFolder.getAbsolutePath() + " should exists");

    this.sourceFolder = sourceFolder;
    this.destinationFolder = destinationFolder;
  }

  public void generate() throws Exception {
    Paths.get(destinationFolder.getAbsolutePath(), APPLICATION.value()).toFile().mkdir();

    copyDirectory(CLASSES.value(), CLASSES.value(), true, false);
    copyDirectory(TEST_CLASSES.value(), CLASSES.value(), false, false);
    // TODO MU-1080
    copyDirectory(MULE.value(), CLASSES.value(), true, false);
    copyDirectory(TEST_MULE.value(), MULE.value(), true, false);
    copyDirectory(META_INF.value(), META_INF.value(), true, false);
    copyDirectory(REPOSITORY.value(), REPOSITORY.value(), false, true);
  }

  private void copyDirectory(String src, String dst, Boolean checkSrc, Boolean doNotCopyIfEmpty) throws Exception {
    Path sourcePath = Paths.get(sourceFolder.getAbsolutePath(), src);
    Path destinationPath = Paths.get(destinationFolder.getAbsolutePath(), APPLICATION.value(), dst);

    String srcAbsolutePath = sourcePath.toAbsolutePath().toString();
    String dstAbsolutePath = destinationPath.toAbsolutePath().toString();


    if (checkSrc && !sourcePath.toFile().exists()) {
      throw new IllegalArgumentException("The source folder " + srcAbsolutePath + " should exists");
    }
    if (!checkSrc && !sourcePath.toFile().exists()) {
      return;
    }

    try {
      if (doNotCopyIfEmpty && FileUtils.listFiles(sourcePath.toFile(), INSTANCE, INSTANCE).isEmpty()) {
        return;
      }
      FileUtils.copyDirectory(sourcePath.toFile(), destinationPath.toFile());
    } catch (IOException e) {
      throw new Exception("Fail to copy content from " + srcAbsolutePath + " to " + dstAbsolutePath, e);
    }
  }
}
