/*
 * Copyright (c) 2015 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.remote;


import static java.util.Collections.singletonList;
import static org.mule.munit.remote.FolderNames.*;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileWriter;
import java.io.IOException;
import java.net.URISyntaxException;
import java.nio.charset.Charset;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.List;
import java.util.Properties;

import org.apache.commons.cli.ParseException;
import org.apache.commons.lang3.exception.ExceptionUtils;
import org.mule.munit.common.exception.MunitError;
import org.mule.munit.common.properties.MUnitUserPropertiesManager;
import org.mule.munit.common.protocol.listeners.RemoteRunEventListener;
import org.mule.munit.common.protocol.listeners.RunEventListener;
import org.mule.munit.common.protocol.listeners.RunEventListenerContainer;
import org.mule.munit.common.util.FileUtils;
import org.mule.munit.remote.api.client.RunnerClient;
import org.mule.munit.remote.classloading.ClassLoaderUtils;
import org.mule.munit.remote.config.NotifierConfiguration;
import org.mule.munit.remote.config.RunConfiguration;
import org.mule.munit.remote.config.RunConfigurationParser;
import org.mule.munit.remote.container.ContainerFactory;
import org.mule.munit.remote.notifier.NotifierReflectionFactory;
import org.mule.runtime.api.deployment.meta.MuleApplicationModel;
import org.mule.runtime.api.deployment.meta.MuleApplicationModel.MuleApplicationModelBuilder;
import org.mule.runtime.api.deployment.meta.MuleArtifactLoaderDescriptorBuilder;
import org.mule.runtime.api.deployment.persistence.MuleApplicationModelJsonSerializer;
import org.mule.runtime.module.embedded.api.EmbeddedContainer;


/**
 * This is the main entry point to run a set of MUnit Test Suite files.
 *
 *
 * If could be run from any command line on its own VM provided it receives the proper parameters defined in @link
 * {@link RunConfigurationParser}.
 *
 * The final goal of the class is to run each MUnit Test Suite file provided and report back the run results and the coverage
 * results.
 *
 * @author Mulesoft Inc.
 * @since 2.0.0
 */
public class RemoteRunner {

  public static final String MUNIT_SERVER_PORT = "munit.server.port";
  public static final String SYSTEM_PROPERTIES_FILE = "munit.system.properties.file";
  public static final String MUNIT_DEBUG_LOG_CLASSPATH = "munit.debug.log.classpath";
  private static final String CONTAINER_START_FAILURE = "org.mule.runtime.deployment.model.api.DeploymentInitException";
  public static final String MULE_APPLICATION_JSON = "mule-application.json";

  private ContainerFactory containerFactory;
  private MuleApplicationModelBuilder muleApplicationModelBuilder = null;

  public static void main(String args[]) throws ParseException, IOException, URISyntaxException {
    loadSystemPropertiesFileIfPresent();
    RunConfiguration runConfig = new RunConfigurationParser().parse(args);
    RemoteRunner runner = new RemoteRunner();
    runner.setContainerFactory(new ContainerFactory());
    runner.run(runConfig);
    shutDown();
  }

  public void setContainerFactory(ContainerFactory containerFactory) {
    this.containerFactory = containerFactory;
  }

  public void run(RunConfiguration runConfig) throws IOException, URISyntaxException {
    System.out.println("[" + this.getClass().getName() + "]" + "Run Started");

    logClassPathIfNecessary();

    RemoteRunEventListener listener = buildRunnerListener(runConfig.getNotifierConfigurations());
    int munitRunnerPort = Integer.parseInt(System.getProperty(MUNIT_SERVER_PORT));
    listener.notifyRunStart();
    try {
      for (String munitSuite : runConfig.getSuitePaths()) {
        runSuite(runConfig, listener, munitSuite, munitRunnerPort);
      }
    } catch (Throwable e) {
      e.printStackTrace();
      listener.notifyUnexpectedError(ExceptionUtils.getStackTrace(e));
    }
    listener.notifyRunFinish();

    System.out.println("[" + this.getClass().getName() + "]" + "Done");
  }

  protected void runSuite(RunConfiguration runConfig, RemoteRunEventListener listener, String suite, int port) throws Exception {

    updateMuleApplicationJson(suite, runConfig.getContainerConfiguration().getMunitWorkingDirectoryPath());

    EmbeddedContainer container = containerFactory.createContainer(runConfig.getContainerConfiguration());
    try {
      container.start();

      RunnerClient runnerClient = buildRunnerClient(listener, port);
      runnerClient.sendSuiteRunInfo(runConfig.getRunToken(), suite, runConfig.getTestNames(), runConfig.getTags());
      runnerClient.receiveAndNotify();

      container.stop();
    } catch (RuntimeException e) {
      if (isContainerFailure(e)) {
        listener.notifyContainerFailure(suite, ExceptionUtils.getStackTrace(e));
        e.printStackTrace();
      } else {
        throw e;
      }
    }

    // coverageManager.stopCoverageServer();

    // calculateCoverageIfApplicable(coverageManager, listener);
  }

  private void updateMuleApplicationJson(String suitePath, String munitWorkingDirectoryPath) throws IOException {
    Path relativeSuitePath = Paths.get(MUNIT.value(), suitePath);

    Path applicationJsonPath =
        Paths.get(munitWorkingDirectoryPath, APPLICATION.value(), META_INF.value(), MULE_ARTIFACT.value(), MULE_APPLICATION_JSON);
    MuleApplicationModelBuilder builder = getApplicationModelBuilder(applicationJsonPath.toFile());
    builder.setConfigs(singletonList(relativeSuitePath.toString()));

    String applicationDescriptorContent = new MuleApplicationModelJsonSerializer().serialize(builder.build());
    try (FileWriter fileWriter = new FileWriter(applicationJsonPath.toAbsolutePath().toString())) {
      fileWriter.write(applicationDescriptorContent);
    } catch (IOException e) {
      throw new MunitError("Fail to update: " + applicationJsonPath.toAbsolutePath().toString(), e);
    }
  }

  private MuleApplicationModelBuilder getApplicationModelBuilder(File applicationJsonFile) throws IOException {
    if (muleApplicationModelBuilder == null) {
      MuleApplicationModel muleApplicationModel = getMuleApplicationModel(applicationJsonFile);

      MuleApplicationModelBuilder builder = new MuleApplicationModelBuilder();
      builder.setMinMuleVersion(muleApplicationModel.getMinMuleVersion());
      muleApplicationModel.getDomain().ifPresent(builder::setDomain);
      builder.setRedeploymentEnabled(muleApplicationModel.isRedeploymentEnabled());
      builder.setName(muleApplicationModel.getName());

      MuleArtifactLoaderDescriptorBuilder muleArtifactLoaderDescriptorBuilder = builder.withClassLoaderModelDescriber();
      if (muleApplicationModel.getClassLoaderModelLoaderDescriptor().isPresent()) {
        muleArtifactLoaderDescriptorBuilder.setId(muleApplicationModel.getClassLoaderModelLoaderDescriptor().get().getId());
      } else {
        throw new MunitError("Fail to read " + applicationJsonFile.getAbsolutePath()
            + ". Classloader model descriptor is missing.");
      }

      builder.withBundleDescriptorLoader(muleApplicationModel.getBundleDescriptorLoader());
      builder.withBundleDescriptorLoader(muleArtifactLoaderDescriptorBuilder.build());

      muleApplicationModelBuilder = builder;
    }
    return muleApplicationModelBuilder;
  }

  private MuleApplicationModel getMuleApplicationModel(File applicationJsonFile) throws IOException {
    return new MuleApplicationModelJsonSerializer()
        .deserialize(FileUtils.readFileToString(applicationJsonFile, Charset.defaultCharset()));
  }

  protected RunnerClient buildRunnerClient(RemoteRunEventListener listener, int port) throws IOException {
    return new RunnerClient(port, listener);
  }

  protected boolean isContainerFailure(RuntimeException e) {
    return e.getCause() != null && CONTAINER_START_FAILURE.equals(e.getCause().getClass().getCanonicalName());
  }

  protected RemoteRunEventListener buildRunnerListener(List<NotifierConfiguration> configurations) throws IOException {
    List<RunEventListener> listeners = new NotifierReflectionFactory().createNotifiers(configurations);
    RunEventListenerContainer container = new RunEventListenerContainer();
    listeners.forEach(container::addNotificationListener);
    return container;
  }

  private void logClassPathIfNecessary() {
    if (Boolean.valueOf(System.getProperty(MUNIT_DEBUG_LOG_CLASSPATH, "false"))) {
      System.out.println("[" + this.getClass().getName() + "]" + "logging classpath ...");
      new ClassLoaderUtils().getClassPath().forEach(System.out::println);
      System.out.println("[" + this.getClass().getName() + "]" + "logging classpath DONE");
    }
  }

  // TODO MU-906
  // private CoverageManager buildCoverageManager(RunConfiguration runConfig, ClassLoaderHolder classLoaderHolder) {
  // boolean shouldCalculateCoverage = runConfig.isRunCoverage();
  // File projectPath = runConfig.getAppDir();
  // CoverageManager coverageManager;
  // if (!shouldCalculateCoverage) {
  // coverageManager = new CoverageManager(-1, "", false, projectPath, classLoaderHolder);
  // } else {
  // int coveragePort = runConfig.getCoveragePort();
  // System.setProperty(CoveragePlugin.COVERAGE_PORT_PROPERTY, String.valueOf(coveragePort));
  // String applicationResources = runConfig.getApplicationPaths();
  //
  // System.out.format("Coverage port: %d resources: %s \n", coveragePort, applicationResources);
  // coverageManager = new CoverageManager(coveragePort, applicationResources, true, projectPath, classLoaderHolder);
  // coverageManager.setIgnoreFlows(runConfig.getIgnoreFlows());
  // }
  // return coverageManager;
  // }

  // TODO MU-906
  // private void calculateCoverageIfApplicable(CoverageManager coverageManager, RemoteRunnerEventListener listener) {
  // ApplicationCoverageReport coverageReport = coverageManager.generateCoverageReport();
  // String coverageReportJson = new JSONObject(coverageReport).toString();
  // listener.notifyCoverageReport(coverageReportJson);
  // }

  private static void loadSystemPropertiesFileIfPresent() {
    String filePath = System.getProperty(SYSTEM_PROPERTIES_FILE);
    try {
      if (filePath != null) {
        Properties properties = new Properties();
        properties.load(new FileInputStream(new File(filePath)));
        MUnitUserPropertiesManager.loadPropertiesToSystem(properties);
      }
    } catch (IOException e) {
      // TODO handle this properly
    }
  }

  private static void shutDown() {
    System.exit(0);
  }
}
