package org.mule.munit.remote.api.client;

import com.google.common.collect.ImmutableMap;
import com.google.gson.Gson;
import org.apache.commons.lang3.StringUtils;
import org.mule.munit.common.protocol.listeners.RemoteRunEventListener;
import org.mule.munit.common.protocol.listeners.RunEventListener;
import org.mule.munit.common.protocol.listeners.RunEventListenerContainer;
import org.mule.munit.common.protocol.message.RunMessage;
import org.mule.munit.common.protocol.message.RunMessageParser;

import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.net.Socket;
import java.util.Collection;
import java.util.Set;
import java.util.stream.Collectors;

import static org.mule.munit.common.protocol.message.MessageField.*;
import static org.mule.munit.common.protocol.message.MessageID.RUN_SUITE;

/**
 * Sends messages to the munit-runner plugin.
 *
 * @author Mulesoft Inc.
 * @since 2.0.0
 */
public class RunnerClient {

  private ObjectInputStream in;
  private ObjectOutputStream out;
  private boolean suiteRunning = true;
  private RunMessageParser parser;

  public RunnerClient(int port, RemoteRunEventListener runnerEventListener) throws IOException {
    Socket requestSocket = new Socket("localhost", port);
    in = new ObjectInputStream(requestSocket.getInputStream());
    out = new ObjectOutputStream(requestSocket.getOutputStream());
    parser = buildParser(runnerEventListener);
  }

  public void sendSuiteRunInfo(String runToken, String suite, Set<String> testNames, Set<String> tags) throws IOException {
    send(new Gson().toJson(new RunMessage(RUN_SUITE, ImmutableMap.of(RUN_TOKEN_KEY, runToken,
                                                                     MUNIT_SUITE_KEY, suite,
                                                                     TEST_NAMES_KEY, collectionToString(testNames),
                                                                     TAGS_KEY, collectionToString(tags)))));
  }

  public void receiveAndNotify() throws IOException, ClassNotFoundException {
    do {
      String message = (String) in.readObject();
      parser.parseAndNotify(message);
    } while (suiteRunning);
  }

  private void finish() {
    suiteRunning = false;
  }

  private void send(String message) throws IOException {
    out.writeObject(message);
    out.flush();
  }

  private String collectionToString(Collection<String> collection) {
    return collection == null ? StringUtils.EMPTY : collection.stream().collect(Collectors.joining(SEPARATOR_TOKEN));
  }

  private RunMessageParser buildParser(RemoteRunEventListener runnerEventListener) {
    RunEventListenerContainer runnerEventListenerContainer = new RunEventListenerContainer();
    runnerEventListenerContainer.addNotificationListener((RunEventListener) runnerEventListener);
    runnerEventListenerContainer.addNotificationListener(new SuiteFinishedListener());

    return new RunMessageParser(runnerEventListenerContainer);
  }

  private class SuiteFinishedListener implements RunEventListener {

    @Override
    public void notifySuiteUnexpectedError(String stackTrace) {
      notifyUnexpectedError(stackTrace);
    }

    @Override
    public void notifySuiteEnd(String suite, long elapsedTime) {
      finish();
    }

    @Override
    public void notifyUnexpectedError(String stackTrace) {
      finish();
    }
  }

}
