/*
 * Copyright (c) 2015 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.remote.container;

import static org.mule.maven.client.api.model.MavenConfiguration.newMavenConfigurationBuilder;
import static org.mule.maven.client.api.model.RemoteRepository.newRemoteRepositoryBuilder;
import static org.mule.munit.remote.FolderNames.APPLICATION;
import static org.mule.munit.remote.FolderNames.CONTAINER;

import java.io.File;
import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URL;
import java.nio.file.Paths;

import org.apache.commons.lang3.StringUtils;

import org.mule.maven.client.api.model.MavenConfiguration;
import org.mule.munit.remote.config.ContainerConfiguration;
import org.mule.runtime.module.embedded.api.ApplicationConfiguration;
import org.mule.runtime.module.embedded.api.DeploymentConfiguration;
import org.mule.runtime.module.embedded.api.EmbeddedContainer;
import org.mule.runtime.module.embedded.api.Product;

/**
 * It knows how to build a embedded container, used to start Mule programmatically
 */
public class ContainerFactory {

  protected static final String MAVEN_CENTRAL_ID = "maven-central";
  protected static final String MULESOFT_PUBLIC_ID = "mulesoft-public";

  protected static final String MAVEN_CENTRAL_URL = "https://repo.maven.apache.org/maven2/";
  protected static final String MULESOFT_PUBLIC_URL = "https://repository.mulesoft.org/nexus/content/repositories/public";

  public EmbeddedContainer createContainer(ContainerConfiguration configuration) throws IOException {
    URL containerBaseFolder = Paths.get(configuration.getMunitWorkingDirectoryPath(), CONTAINER.value()).toFile().toURI().toURL();

    String runtimeVersion = configuration.getRuntimeId();
    String product = configuration.getProduct();
    File log4JConfigurationFile = getLog4JConfigurationFile(configuration);
    DeploymentConfiguration deploymentConfiguration = DeploymentConfiguration.builder().withTestDependenciesEnabled(true).build();

    ApplicationConfiguration applicationConfiguration = ApplicationConfiguration.builder()
        .withApplicationLocation(Paths.get(configuration.getMunitWorkingDirectoryPath(), APPLICATION.value()).toFile())
        .withDeploymentConfiguration(deploymentConfiguration)
        .build();

    return EmbeddedContainer.builder()
        .withMuleVersion(runtimeVersion)
        .withProduct(Product.valueOf(product))
        .withLog4jConfigurationFile(log4JConfigurationFile.toURI())
        .withContainerBaseFolder(containerBaseFolder)
        .withApplicationConfiguration(applicationConfiguration)
        .withMavenConfiguration(getMavenConfiguration(configuration.getMavenConfiguration()))
        .build();

  }

  protected File getLog4JConfigurationFile(ContainerConfiguration runConfig) {
    String log4JConfigurationFilePath = runConfig.getLog4JConfigurationFilePath();
    if (StringUtils.isNotBlank(log4JConfigurationFilePath)) {
      File log4JConfigurationFile = Paths.get(log4JConfigurationFilePath).toFile();
      if (log4JConfigurationFile != null && log4JConfigurationFile.exists()) {
        return log4JConfigurationFile;
      }
    }
    return null;
  }

  protected MavenConfiguration getMavenConfiguration(org.mule.munit.remote.config.MavenConfiguration configuration)
      throws MalformedURLException {
    MavenConfiguration.MavenConfigurationBuilder builder = newMavenConfigurationBuilder();

    if (configuration.getMavenRepositoryDirectoryPath() != null) {
      File mavenRepository = Paths.get(configuration.getMavenRepositoryDirectoryPath()).toFile();
      if (mavenRepository.exists()) {
        builder.withLocalMavenRepositoryLocation(mavenRepository);
      }
    }

    if (configuration.getSettingsXmlFilePath() != null) {
      File settingsXml = Paths.get(configuration.getSettingsXmlFilePath()).toFile();
      if (settingsXml.exists()) {
        builder.withUserSettingsLocation(settingsXml);
      }
    }

    if (configuration.getGlobalSettingsXmlFilePath() != null) {
      File globalSettingsXml = Paths.get(configuration.getGlobalSettingsXmlFilePath()).toFile();
      if (globalSettingsXml.exists()) {
        builder.withGlobalSettingsLocation(globalSettingsXml);
      }
    }

    if (configuration.getForcePolicyUpdate()) {
      builder.withForcePolicyUpdateNever(false);
    } else {
      builder.withForcePolicyUpdateNever(true);
    }

    addDefaultRepositories(builder);
    return builder.build();
  }

  protected void addDefaultRepositories(MavenConfiguration.MavenConfigurationBuilder builder) throws MalformedURLException {
    builder.withRemoteRepository(
                                 newRemoteRepositoryBuilder()
                                     .withId(MAVEN_CENTRAL_ID)
                                     .withUrl(new URL(MAVEN_CENTRAL_URL))
                                     .build());
    builder.withRemoteRepository(
                                 newRemoteRepositoryBuilder()
                                     .withId(MULESOFT_PUBLIC_ID)
                                     .withUrl(new URL(MULESOFT_PUBLIC_URL))
                                     .build());
  }

}
