/*
 * Copyright (c) 2015 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.remote.notifier;

import org.apache.commons.lang3.StringUtils;
import org.mule.munit.common.protocol.listeners.RunEventListener;
import org.mule.munit.common.protocol.notifiers.RunMessageNotifier;
import org.mule.munit.remote.config.NotifierConfiguration;
import org.mule.munit.remote.config.NotifierParameter;

import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.List;
import java.util.stream.Collectors;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Preconditions.checkNotNull;

/**
 * It knows how to generate notifiers using reflection based on a notifier configuration
 */
public class NotifierReflectionFactory {

  public static final String SET_PREFIX = "set";

  public List<RunEventListener> createNotifiers(List<NotifierConfiguration> notifierConfigurations) {
    checkNotNull(notifierConfigurations, "The configurations can not be null");
    checkArgument(notifierConfigurations.size() >= 1, "There must be at least one notifier configuration");

    return notifierConfigurations.stream().map(this::createNotifierInstance).collect(Collectors.toList());
  }

  private RunEventListener createNotifierInstance(NotifierConfiguration nc) {
    try {
      RunEventListener listener = (RunEventListener) Class.forName(nc.getClazz()).newInstance();
      configureNotifier(nc.getParameters(), listener);
      initNotifier(listener);
      return listener;
    } catch (InstantiationException | IllegalAccessException | ClassNotFoundException e) {
      throw new RuntimeException("Fail to create instance for notifier: " + nc.getClazz(), e);
    }
  }

  private void configureNotifier(List<NotifierParameter> parameters, RunEventListener listener) {
    for (NotifierParameter parameter : parameters) {
      Method method = getMethod(listener, parameter);
      method.setAccessible(true);
      try {
        method.invoke(listener, parameter.getValue());
      } catch (IllegalAccessException | InvocationTargetException e) {
        throw new RuntimeException("Fail to configure instance of notifier: " + listener.getClass().getCanonicalName(),
                                   e);
      }
    }
  }

  private Method getMethod(RunEventListener listener, NotifierParameter parameter) {
    String setMethod = SET_PREFIX + StringUtils.capitalize(parameter.getName());
    try {
      return listener.getClass().getMethod(setMethod, Class.forName(parameter.getType()));
    } catch (NoSuchMethodException | ClassNotFoundException e) {
      throw new RuntimeException(
                                 "Fail to get setter method: " + setMethod + " in class "
                                     + listener.getClass().getCanonicalName(),
                                 e);
    }
  }

  private void initNotifier(RunEventListener listener) {
    try {
      ((RunMessageNotifier) listener).init();
    } catch (Exception e) {
      throw new RuntimeException("Fail to init instance of notifier: " + listener.getClass().getCanonicalName(), e);
    }
  }
}
