/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.remote;

import static com.google.common.base.Preconditions.checkArgument;
import static org.apache.commons.io.filefilter.TrueFileFilter.INSTANCE;
import static org.mule.munit.remote.FolderNames.CLASSES;
import static org.mule.munit.remote.FolderNames.META_INF;
import static org.mule.munit.remote.FolderNames.MULE_ARTIFACT;
import static org.mule.munit.remote.FolderNames.TEST_CLASSES;
import static org.mule.munit.remote.FolderNames.TEST_MULE;

import java.io.File;
import java.io.IOException;
import java.nio.file.Path;

import org.apache.commons.io.FileUtils;
import org.apache.commons.lang3.StringUtils;

/**
 * It create a structure similar to that of an exploded application based on a source folder containing all the required folders
 *
 * @author Mulesoft Inc.
 * @since 2.0.0
 */
public class ApplicationStructureGenerator {

  private static final String CLASSLOADER_MODEL_JSON = "classloader-model.json";

  private Path originFolderPath;
  private Path destinationFolderPath;

  public ApplicationStructureGenerator(File sourceFolder, File destinationFolder, String applicationName) {
    checkArgument(sourceFolder.exists(), "The source folder " + sourceFolder.getAbsolutePath() + " should exists");
    checkArgument(destinationFolder.exists(), "The destination folder " + destinationFolder.getAbsolutePath() + " should exists");
    checkArgument(StringUtils.isNotBlank(applicationName), "The application name must not be null nor empty");

    this.originFolderPath = sourceFolder.toPath();
    this.destinationFolderPath = destinationFolder.toPath().resolve(applicationName);
  }

  public void generate() throws Exception {
    destinationFolderPath.toFile().mkdir();

    copyDirectory(CLASSES.value(), StringUtils.EMPTY, true, false);
    copyDirectory(TEST_CLASSES.value(), StringUtils.EMPTY, false, false);
    copyDirectory(TEST_MULE.value(), StringUtils.EMPTY, true, false);
    copyDirectory(META_INF.value(), META_INF.value(), true, false);

    removeClassLoaderModelDescriptorIfPresent();
  }

  private void copyDirectory(String src, String dst, Boolean checkSrc, Boolean doNotCopyIfEmpty) throws Exception {
    Path sourcePath = originFolderPath.resolve(src);
    Path destinationPath = destinationFolderPath.resolve(dst);

    String srcAbsolutePath = sourcePath.toAbsolutePath().toString();
    String dstAbsolutePath = destinationPath.toAbsolutePath().toString();


    if (checkSrc && !sourcePath.toFile().exists()) {
      throw new IllegalArgumentException("The source folder " + srcAbsolutePath + " should exists");
    }
    if (!checkSrc && !sourcePath.toFile().exists()) {
      return;
    }

    try {
      if (doNotCopyIfEmpty && FileUtils.listFiles(sourcePath.toFile(), INSTANCE, INSTANCE).isEmpty()) {
        return;
      }
      FileUtils.copyDirectory(sourcePath.toFile(), destinationPath.toFile());
    } catch (IOException e) {
      throw new Exception("Fail to copy content from " + srcAbsolutePath + " to " + dstAbsolutePath, e);
    }
  }

  private void removeClassLoaderModelDescriptorIfPresent() {
    Path destinationPath = destinationFolderPath
        .resolve(META_INF.value()).resolve(MULE_ARTIFACT.value()).resolve(CLASSLOADER_MODEL_JSON);

    if (destinationPath.toFile().exists()) {
      destinationPath.toFile().delete();
    }
  }
}
