/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.remote;


import static java.util.Collections.emptySet;
import static java.util.Collections.singletonList;
import static java.util.Optional.ofNullable;
import static org.mule.munit.remote.FolderNames.META_INF;
import static org.mule.munit.remote.FolderNames.MULE_ARTIFACT;
import static org.mule.munit.remote.FolderNames.MUNIT;
import static org.mule.munit.remote.MuleArtifactHandler.MULE_ARTIFACT_JSON;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.net.URISyntaxException;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.List;
import java.util.Optional;
import java.util.Properties;

import org.apache.commons.cli.ParseException;
import org.apache.commons.lang3.exception.ExceptionUtils;
import org.mule.munit.common.properties.MUnitUserPropertiesManager;
import org.mule.munit.common.protocol.listeners.RemoteRunEventListener;
import org.mule.munit.common.protocol.listeners.RunEventListener;
import org.mule.munit.common.protocol.listeners.RunEventListenerContainer;
import org.mule.munit.remote.api.client.RunnerClient;
import org.mule.munit.remote.api.configuration.CoverageConfiguration;
import org.mule.munit.remote.api.configuration.NotifierConfiguration;
import org.mule.munit.remote.api.configuration.RunConfiguration;
import org.mule.munit.remote.api.configuration.RunConfigurationParser;
import org.mule.munit.remote.classloading.ClassLoaderUtils;
import org.mule.munit.remote.container.ContainerFactory;
import org.mule.munit.remote.container.IllegalPortDefinitionException;
import org.mule.munit.remote.container.MunitRunnerPortProvider;
import org.mule.munit.remote.coverage.CoverageManager;
import org.mule.munit.remote.coverage.report.model.ApplicationCoverageReport;
import org.mule.munit.remote.exception.DeploymentException;
import org.mule.munit.remote.exception.DeploymentExceptionThrower;
import org.mule.munit.remote.notifier.NotifierReflectionFactory;
import org.mule.runtime.module.embedded.api.ArtifactConfiguration;
import org.mule.runtime.module.embedded.api.DeploymentConfiguration;
import org.mule.runtime.module.embedded.api.DeploymentService;
import org.mule.runtime.module.embedded.api.EmbeddedContainer;

import com.google.gson.Gson;


/**
 * This is the main entry point to run a set of MUnit Test Suite files.
 *
 *
 * If could be run from any command line on its own VM provided it receives the proper parameters defined in @link
 * {@link RunConfigurationParser}.
 *
 * The final goal of the class is to run each MUnit Test Suite file provided and report back the run results and the coverage
 * results.
 *
 * @author Mulesoft Inc.
 * @since 2.0.0
 */
public class RemoteRunner {

  public static final String COVERAGE_SEND_ALL_LOCATIONS_PROPERTY = "coverage.send.all.locations";

  public static final String MUNIT_SERVER_PORT = "munit.server.port";
  public static final String SYSTEM_PROPERTIES_FILE = "munit.system.properties.file";
  public static final String MUNIT_DEBUG_LOG_CLASSPATH = "munit.debug.log.classpath";

  private ContainerFactory containerFactory;

  public static void main(String args[]) throws ParseException, IOException, URISyntaxException {
    loadSystemPropertiesFileIfPresent();
    RunConfiguration runConfig = new RunConfigurationParser().parse(args);
    RemoteRunner runner = new RemoteRunner();
    runner.setContainerFactory(new ContainerFactory());
    runner.run(runConfig);
    shutDown();
  }

  public void setContainerFactory(ContainerFactory containerFactory) {
    this.containerFactory = containerFactory;
  }

  public void run(RunConfiguration runConfig) throws IOException, URISyntaxException {
    System.out.println("[" + this.getClass().getName() + "]" + "Run Started");
    logClassPathIfNecessary();

    RemoteRunEventListener listener = buildRunnerListener(runConfig.getNotifierConfigurations());
    listener.notifyRunStart();

    Integer munitRunnerPort;
    try {
      munitRunnerPort = new MunitRunnerPortProvider().getPort();
    } catch (IllegalPortDefinitionException e) {
      listener.notifyUnexpectedError(ExceptionUtils.getStackTrace(e));
      listener.notifyRunFinish();
      System.out.println("[" + this.getClass().getName() + "]" + "Done");
      return;
    }

    EmbeddedContainer container = containerFactory.createContainer(runConfig.getContainerConfiguration());
    DeploymentConfiguration deploymentConfiguration = DeploymentConfiguration.builder().testDependenciesEnabled(true).build();
    MuleArtifactHandler muleArtifactHandler = buildMuleArtifactHandler(runConfig);
    CoverageManager coverageManager = buildCoverageManager(runConfig.getCoverageConfiguration());

    try {
      container.start();
      coverageManager.startCoverageServer();
      for (String suitePath : runConfig.getSuitePaths()) {
        boolean success =
            runSuite(container, runConfig, muleArtifactHandler, deploymentConfiguration, listener, suitePath, munitRunnerPort);
        if (!success && runConfig.isSkipAfterFailure()) {
          break;
        }
      }

      if (coverageManager.shouldCalculateCoverage()) {
        deployAndUndeployFullApplication(container, runConfig, muleArtifactHandler, deploymentConfiguration);
      }

    } catch (Throwable e) {
      e.printStackTrace();
      listener.notifyUnexpectedError(ExceptionUtils.getStackTrace(e));
    } finally {
      container.stop();
    }

    coverageManager.stopCoverageServer();
    sendCoverageReport(coverageManager, listener);

    listener.notifyRunFinish();
    System.out.println("[" + this.getClass().getName() + "]" + "Done");
  }

  protected boolean runSuite(EmbeddedContainer container, RunConfiguration runConfig, MuleArtifactHandler muleArtifactHandler,
                             DeploymentConfiguration deploymentConfiguration, RemoteRunEventListener listener, String suitePath,
                             int port) {
    String applicationName = runConfig.getProjectName();
    String munitWorkingDirectoryPath = runConfig.getContainerConfiguration().getMunitWorkingDirectoryPath();
    try {
      updateMuleArtifactJson(suitePath, muleArtifactHandler);

      deployApplication(applicationName, deploymentConfiguration, munitWorkingDirectoryPath, container);

      RunnerClient runnerClient = buildRunnerClient(listener, port);
      runnerClient.sendSuiteRunInfo(runConfig.getRunToken(), suitePath, runConfig.getTestNames(), runConfig.getTags());
      return runnerClient.receiveAndNotify();

    } catch (RuntimeException | IOException | ClassNotFoundException | DeploymentException e) {
      listener.notifyContainerFailure(suitePath, ExceptionUtils.getStackTrace(e));
      return false;
    } finally {
      undeployApplication(container, applicationName);
    }
  }

  protected void deployAndUndeployFullApplication(EmbeddedContainer container, RunConfiguration runConfig,
                                                  MuleArtifactHandler muleArtifactHandler,
                                                  DeploymentConfiguration deploymentConfiguration) {
    String applicationName = runConfig.getProjectName();
    String munitWorkingDirectoryPath = runConfig.getContainerConfiguration().getMunitWorkingDirectoryPath();
    try {
      muleArtifactHandler.restoreOriginal(true);
      System.setProperty(COVERAGE_SEND_ALL_LOCATIONS_PROPERTY, "true");

      System.out.println("[ COVERAGE ] Deploying complete application...");
      deployApplication(applicationName, deploymentConfiguration, munitWorkingDirectoryPath, container);

    } catch (RuntimeException | IOException | DeploymentException e) {
      System.out.println("[ COVERAGE ] Deploying complete application failed");
    } finally {
      undeployApplication(container, applicationName);
      System.out.println("[ COVERAGE ] Application undeployed");
      System.clearProperty(COVERAGE_SEND_ALL_LOCATIONS_PROPERTY);
    }
  }

  private void deployApplication(String applicationName, DeploymentConfiguration deploymentConfiguration,
                                 String munitWorkingDirectoryPath, EmbeddedContainer container)
      throws DeploymentException {
    Path artifactLocation = Paths.get(munitWorkingDirectoryPath, applicationName);
    ArtifactConfiguration artifactConfiguration = getArtifactConfiguration(artifactLocation, deploymentConfiguration);
    DeploymentService deploymentService = container.getDeploymentService();

    try {
      deploymentService.deployApplication(artifactConfiguration);
    } catch (RuntimeException e) {
      DeploymentExceptionThrower.throwIfMatches(e);
    }
  }

  private void undeployApplication(EmbeddedContainer container, String applicationName) {
    container.getDeploymentService().undeployApplication(applicationName);
  }

  private ArtifactConfiguration getArtifactConfiguration(Path artifactLocation, DeploymentConfiguration deploymentConfiguration) {
    return ArtifactConfiguration.builder()
        .deploymentConfiguration(deploymentConfiguration)
        .artifactLocation(artifactLocation.toFile())
        .build();
  }

  private MuleArtifactHandler buildMuleArtifactHandler(RunConfiguration runConfig) {
    String applicationName = runConfig.getProjectName();
    String munitWorkingDirectoryPath = runConfig.getContainerConfiguration().getMunitWorkingDirectoryPath();

    Path applicationJsonPath = Paths.get(munitWorkingDirectoryPath, applicationName).resolve(META_INF.value())
        .resolve(MULE_ARTIFACT.value()).resolve(MULE_ARTIFACT_JSON);

    return new MuleArtifactHandler(applicationJsonPath);
  }

  private void updateMuleArtifactJson(String suitePath, MuleArtifactHandler muleArtifactHandler)
      throws IOException {
    Path relativeSuitePath = Paths.get(MUNIT.value(), suitePath);
    muleArtifactHandler.updateConfigPathAndReDeployment(singletonList(relativeSuitePath.toString()), true);
  }

  protected RunnerClient buildRunnerClient(RemoteRunEventListener listener, int port) throws IOException {
    return new RunnerClient(port, listener);
  }

  protected RemoteRunEventListener buildRunnerListener(List<NotifierConfiguration> configurations) throws IOException {
    List<RunEventListener> listeners = new NotifierReflectionFactory().createNotifiers(configurations);
    RunEventListenerContainer container = new RunEventListenerContainer();
    listeners.forEach(container::addNotificationListener);
    return container;
  }

  private void logClassPathIfNecessary() {
    if (Boolean.valueOf(System.getProperty(MUNIT_DEBUG_LOG_CLASSPATH, "false"))) {
      System.out.println("[" + this.getClass().getName() + "]" + "logging classpath ...");
      new ClassLoaderUtils().getClassPath().forEach(System.out::println);
      System.out.println("[" + this.getClass().getName() + "]" + "logging classpath DONE");
    }
  }

  private CoverageManager buildCoverageManager(CoverageConfiguration coverageConfiguration) {
    Integer coverageServerPort = coverageConfiguration.getCoveragePort();
    Boolean randomizeCoveragePort = coverageConfiguration.isRandomizeCoveragePort();

    CoverageManager coverageManager =
        new CoverageManager(randomizeCoveragePort, coverageServerPort, coverageConfiguration.isRunCoverage(),
                            coverageConfiguration.getSuitePaths());
    coverageManager.setIgnoreFlows(ofNullable(coverageConfiguration.getIgnoredFlowNames()).orElse(emptySet()));
    coverageManager.setIgnoreFiles(ofNullable(coverageConfiguration.getIgnoredFiles()).orElse(emptySet()));
    return coverageManager;
  }

  private void sendCoverageReport(CoverageManager coverageManager, RemoteRunEventListener listener) {
    Optional<ApplicationCoverageReport> coverageReport = coverageManager.generateCoverageReport();
    if (coverageReport.isPresent()) {
      String coverageReportJson = new Gson().toJson(coverageReport.get());
      listener.notifyCoverageReport(coverageReportJson);
    }
  }

  private static void loadSystemPropertiesFileIfPresent() {
    String filePath = System.getProperty(SYSTEM_PROPERTIES_FILE);
    try {
      if (filePath != null) {
        Properties properties = new Properties();
        properties.load(new FileInputStream(new File(filePath)));
        MUnitUserPropertiesManager.loadPropertiesToSystem(properties);
      }
    } catch (IOException e) {
      // TODO handle this properly
    }
  }

  private static void shutDown() {
    System.exit(0);
  }
}
