/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.remote.classloading;

import java.io.File;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.net.MalformedURLException;
import java.net.URL;
import java.net.URLClassLoader;
import java.net.URLDecoder;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.jar.Attributes;
import java.util.jar.Manifest;

/**
 * The class contain utility methods to work with classLoaders
 *
 * @author Mulesoft Inc.
 * @since 2.0.0
 */
public class ClassLoaderUtils extends ClassLoader {

  public static final String META_INF = "META-INF";
  public static final String MANIFEST_MF = "MANIFEST.MF";
  public static final String CLASS_PATH_MANIFEST_ATTR = "Class-Path";
  public static final String STARTER_JAR_FILE_NAME = "munit.classloader.starter.jar.file.name";

  public List<String> getClassPath() {
    List<String> classpath = new ArrayList<>();
    try {
      List<URL> systemUrls = getSystemClassLoaderURLs();
      List<URL> projectUrls = buildProjectClassPathFromManifest();

      systemUrls.stream().map(u -> u.getPath()).forEach(u -> classpath.add(u));
      projectUrls.stream().map(u -> u.getPath()).forEach(u -> classpath.add(u));

    } catch (NoSuchMethodException | InvocationTargetException | IllegalAccessException e) {
      e.printStackTrace();
    }
    return classpath;
  }

  private List<URL> getSystemClassLoaderURLs() throws NoSuchMethodException, InvocationTargetException, IllegalAccessException {

    ClassLoader sysCl = Thread.currentThread().getContextClassLoader();
    Class refClass = URLClassLoader.class;

    Method methodAddUrl = refClass.getDeclaredMethod("getURLs");
    methodAddUrl.setAccessible(true);
    URL[] sysClUrls = (URL[]) methodAddUrl.invoke(sysCl);

    return Arrays.asList(sysClUrls);
  }

  public void enhanceClassLoader() {
    try {
      addUrlsToClassPath(buildProjectClassPathFromManifest());
    } catch (NoSuchMethodException | InvocationTargetException | IllegalAccessException e) {
      e.printStackTrace();
    }
  }

  private void addUrlsToClassPath(List<URL> urls)
      throws NoSuchMethodException, InvocationTargetException, IllegalAccessException {

    ClassLoader sysCl = Thread.currentThread().getContextClassLoader();
    Class refClass = URLClassLoader.class;
    Method methodAddUrl = refClass.getDeclaredMethod("addURL", new Class[] {URL.class});
    methodAddUrl.setAccessible(true);

    for (Iterator it = urls.iterator(); it.hasNext();) {
      URL url = (URL) it.next();
      methodAddUrl.invoke(sysCl, url);
    }
  }

  private List<URL> buildProjectClassPathFromManifest() {
    HashSet<URL> projectClassPath = new LinkedHashSet<>();
    try {
      String manifestJarPath = System.getProperty(STARTER_JAR_FILE_NAME) + "!/" + META_INF + "/" + MANIFEST_MF;

      List<URL> resources = Collections
          .list(Thread.currentThread().getContextClassLoader().getResources(META_INF + "/" + MANIFEST_MF));
      URL manifestURL = resources.stream().filter(resource -> resource.getFile().endsWith(manifestJarPath)).findFirst().get();

      addToProjectClasspath(manifestURL, projectClassPath);
    } catch (IOException e) {
      e.printStackTrace();
    }

    return new ArrayList<>(projectClassPath);
  }

  private void addToProjectClasspath(URL resourceUrl, HashSet<URL> projectClassPath) throws IOException {
    Manifest manifest = new Manifest(resourceUrl.openStream());
    String classpath = (String) manifest.getMainAttributes().get(new Attributes.Name(CLASS_PATH_MANIFEST_ATTR));
    Arrays.asList(classpath.split("file:")).forEach(path -> addToSetOfUrls(path, projectClassPath));
  }

  private void addToSetOfUrls(String path, HashSet<URL> urls) {
    try {
      urls.add(new File(URLDecoder.decode(path.trim(), "UTF-8")).toURI().toURL());
    } catch (MalformedURLException | UnsupportedEncodingException e) {
      e.printStackTrace();
    }
  }
}
