/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.remote.container;

import static org.mule.maven.client.api.model.MavenConfiguration.newMavenConfigurationBuilder;
import static org.mule.maven.client.api.model.RemoteRepository.newRemoteRepositoryBuilder;
import static org.mule.munit.remote.FolderNames.CONTAINER;
import static org.mule.munit.remote.container.BundleDescriptorFactory.buildServerPluginDescriptors;

import java.io.File;
import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.List;

import org.apache.commons.lang3.StringUtils;

import org.mule.maven.client.api.model.BundleDescriptor;
import org.mule.maven.client.api.model.MavenConfiguration;
import org.mule.munit.common.util.IOUtils;
import org.mule.munit.remote.api.configuration.ContainerConfiguration;
import org.mule.munit.remote.api.configuration.ServerPluginConfiguration;
import org.mule.runtime.module.embedded.api.EmbeddedContainer;
import org.mule.runtime.module.embedded.api.Product;


/**
 * It knows how to build a embedded container, used to start Mule programmatically
 *
 * @author Mulesoft Inc.
 * @since 2.0.0
 */
public class ContainerFactory {

  protected static final String MAVEN_CENTRAL_ID = "maven-central";
  protected static final String MULESOFT_PUBLIC_ID = "mulesoft-public";

  protected static final String MAVEN_CENTRAL_URL = "https://repo.maven.apache.org/maven2/";
  protected static final String MULESOFT_PUBLIC_URL = "https://repository.mulesoft.org/nexus/content/repositories/public";

  protected static final String DEFAULT_LOG4J2_CONFIG = "default-log4j2-config.xml";

  public EmbeddedContainer createContainer(ContainerConfiguration configuration) throws IOException {
    Path munitWorkingDirectoryPath = Paths.get(configuration.getMunitWorkingDirectoryPath());
    File containerBaseFolder = munitWorkingDirectoryPath.resolve(CONTAINER.value()).toFile();

    String runtimeVersion = configuration.getRuntimeId();
    String product = configuration.getProduct();
    URI log4JConfiguration = getLog4JConfiguration(configuration);
    BundleDescriptor[] serverPlugins = buildServerPluginDescriptors(configuration.getServerPluginConfigurations());
    loadServerPluginProperties(configuration.getServerPluginConfigurations());

    org.mule.runtime.module.embedded.api.ContainerConfiguration containerConfiguration =
        org.mule.runtime.module.embedded.api.ContainerConfiguration.builder()
            .containerFolder(containerBaseFolder)
            .serverPlugins(serverPlugins)
            .build();

    return EmbeddedContainer.builder()
        .muleVersion(runtimeVersion)
        .product(Product.valueOf(product))
        .log4jConfigurationFile(log4JConfiguration)
        .containerConfiguration(containerConfiguration)
        .mavenConfiguration(getMavenConfiguration(configuration.getMavenConfiguration()))
        .build();
  }

  protected URI getLog4JConfiguration(ContainerConfiguration runConfig) throws IOException {
    String log4JConfigurationFilePath = runConfig.getLog4JConfigurationFilePath();
    if (StringUtils.isNotBlank(log4JConfigurationFilePath)) {
      File log4JConfigurationFile = Paths.get(log4JConfigurationFilePath).toFile();
      if (log4JConfigurationFile != null && log4JConfigurationFile.exists()) {
        return log4JConfigurationFile.toURI();
      }
    }
    return getDefaultLog4jConfiguration();
  }

  private URI getDefaultLog4jConfiguration() throws IOException {
    try {
      return IOUtils.getResourceAsUrl(DEFAULT_LOG4J2_CONFIG, this.getClass()).toURI();
    } catch (URISyntaxException e) {
      throw new IOException("Unable to locate default log4j2 configuration file", e);
    }
  }

  protected void loadServerPluginProperties(List<ServerPluginConfiguration> pluginConfig) {
    if (pluginConfig != null) {
      pluginConfig.stream().filter(config -> config.getProperties() != null)
          .forEach(config -> System.getProperties().putAll(config.getProperties()));
    }
  }

  protected MavenConfiguration getMavenConfiguration(org.mule.munit.remote.api.configuration.MavenConfiguration configuration)
      throws MalformedURLException {
    MavenConfiguration.MavenConfigurationBuilder builder = newMavenConfigurationBuilder();

    if (configuration.getMavenRepositoryDirectoryPath() != null) {
      File mavenRepository = Paths.get(configuration.getMavenRepositoryDirectoryPath()).toFile();
      if (mavenRepository.exists()) {
        builder.localMavenRepositoryLocation(mavenRepository);
      }
    }

    if (configuration.getSettingsXmlFilePath() != null) {
      File settingsXml = Paths.get(configuration.getSettingsXmlFilePath()).toFile();
      if (settingsXml.exists()) {
        builder.userSettingsLocation(settingsXml);
      }
    }

    if (configuration.getGlobalSettingsXmlFilePath() != null) {
      File globalSettingsXml = Paths.get(configuration.getGlobalSettingsXmlFilePath()).toFile();
      if (globalSettingsXml.exists()) {
        builder.globalSettingsLocation(globalSettingsXml);
      }
    }
    builder.offlineMode(configuration.getOfflineMode());

    if (configuration.getForcePolicyUpdate()) {
      builder.forcePolicyUpdateNever(false);
    } else {
      builder.forcePolicyUpdateNever(true);
    }

    addDefaultRepositories(builder);
    return builder.build();
  }

  protected void addDefaultRepositories(MavenConfiguration.MavenConfigurationBuilder builder) throws MalformedURLException {
    builder.remoteRepository(newRemoteRepositoryBuilder().id(MAVEN_CENTRAL_ID).url(new URL(MAVEN_CENTRAL_URL))
        .build());
    builder.remoteRepository(newRemoteRepositoryBuilder().id(MULESOFT_PUBLIC_ID).url(new URL(MULESOFT_PUBLIC_URL))
        .build());
  }
}
