/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.remote.coverage;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Preconditions.checkNotNull;
import static java.util.Collections.emptySet;

import java.util.HashSet;
import java.util.Optional;
import java.util.Set;

import org.mule.munit.common.util.FreePortFinder;
import org.mule.munit.plugins.coverage.core.model.CoverageComponentLocation;
import org.mule.munit.plugins.coverage.server.CoverageLocations;
import org.mule.munit.plugins.coverage.server.CoverageLocationsAccumulator;
import org.mule.munit.plugins.coverage.server.CoverageServer;
import org.mule.munit.remote.coverage.report.ApplicationCoverageReportBuilder;
import org.mule.munit.remote.coverage.report.model.ApplicationCoverageReport;

/**
 * It handles the coverage information and know how to process it to generate a report
 *
 * @author Mulesoft Inc.
 * @since 1.0.0
 */
public class CoverageManager implements CoverageLocationsAccumulator {

  private static final int MIN_PORT_NUMBER = 50000;
  private static final int MAX_PORT_NUMBER = 50500;
  private static final String COVERAGE_PORT_PROPERTY = "coverage.port";

  private boolean shouldCalculateCoverage;

  protected Set<String> suitePaths;
  protected Set<String> ignoreFlows = emptySet();
  protected Set<String> ignoreFiles = emptySet();

  protected CoverageServer coverageServer;

  protected Set<CoverageComponentLocation> allLocations = new HashSet<>();
  protected Set<CoverageComponentLocation> coveredLocations = new HashSet<>();

  public CoverageManager(Boolean randomizeCoveragePort, Integer serverPort, Boolean shouldRunCoverage, Set<String> suitePaths) {
    checkNotNull(randomizeCoveragePort, "Randomize coverage port must not null");
    checkNotNull(suitePaths, "The suite path list  must not be null");
    checkArgument(!suitePaths.isEmpty(), "The suite path list  must not be empty");

    this.suitePaths = suitePaths;
    this.shouldCalculateCoverage = shouldRunCoverage;

    if (randomizeCoveragePort) {
      serverPort = new FreePortFinder(MIN_PORT_NUMBER, MAX_PORT_NUMBER).find();
      System.setProperty(COVERAGE_PORT_PROPERTY, serverPort.toString());
    } else {
      checkNotNull(serverPort, "The server port must not null");
    }
    this.coverageServer = new CoverageServer(serverPort, this);
  }

  /**
   * Defines the flows to be ignored for the purposes of coverage calculation
   * 
   * @param ignoreFlows a set of flow names to be ignored
   */
  public void setIgnoreFlows(Set<String> ignoreFlows) {
    checkNotNull(ignoreFlows, "The ignored flows list must not null");
    this.ignoreFlows = ignoreFlows;
  }

  /**
   * Defines the files to be ignored for the purposes of coverage calculation
   *
   * @param ignoreFiles a set of file paths to be ignored
   */
  public void setIgnoreFiles(Set<String> ignoreFiles) {
    checkNotNull(ignoreFiles, "The ignored files list must not null");
    this.ignoreFiles = ignoreFiles;
  }


  public boolean shouldCalculateCoverage() {
    return shouldCalculateCoverage;
  }

  /**
   * Starts the {@link CoverageServer}
   */
  public void startCoverageServer() {
    if (shouldCalculateCoverage) {
      System.out.println("[" + this.getClass().getName() + "]" + " Starting coverage server");
      coverageServer.launch();
    }
  }

  /**
   * Stops the {@link CoverageServer}
   */
  public void stopCoverageServer() {
    if (shouldCalculateCoverage) {
      System.out.println("[" + this.getClass().getName() + "]" + " Shutting down coverage server");
      coverageServer.shutDown();
    }
  }

  @Override
  public synchronized void accumulateCoverageLocations(CoverageLocations coverageLocations) {
    checkNotNull(coverageLocations, "The covered locations must not be null");
    System.out.println("[" + this.getClass().getName() + "]" + "accumulating covered locations");
    if (coverageLocations.isAllLocations()) {
      this.allLocations.addAll(coverageLocations.getCoverageLocations());
    } else {
      this.coveredLocations.addAll(coverageLocations.getCoverageLocations());
    }
  }

  /**
   * Generates the coverage report based on the accumulated covered locations. Returns empty if coverage report is not present
   * 
   * @return an {@link ApplicationCoverageReport}
   */
  public Optional<ApplicationCoverageReport> generateCoverageReport() {
    if (shouldCalculateCoverage) {
      System.out.println("[" + this.getClass().getName() + "]" + "Calculating application coverage");
      ApplicationCoverageReportBuilder reportBuilder = getApplicationCoverageReportBuilder();
      return Optional.of(reportBuilder.build());
    }
    return Optional.empty();
  }

  protected ApplicationCoverageReportBuilder getApplicationCoverageReportBuilder() {
    ApplicationCoverageReportBuilder reportBuilder =
        new ApplicationCoverageReportBuilder(allLocations, coveredLocations, suitePaths);
    reportBuilder.setFlowsToIgnore(ignoreFlows);
    reportBuilder.setFilesToIgnore(ignoreFiles);
    return reportBuilder;
  }

}
