/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.remote.coverage.report.model;

import static com.google.common.base.Preconditions.checkNotNull;

import java.util.ArrayList;
import java.util.List;

/**
 * <p>
 * It holds the coverage information of an application describing:
 * </p>
 * - Overall coverage
 *
 * - Break down by {@link MuleResource}
 *
 * - Break down by {@link MuleFlow}
 *
 * @author Mulesoft Inc.
 * @since 1.0.0
 */
public class ApplicationCoverageReport {

  private Double branchCoverage;
  private List<MuleResource> resources = new ArrayList<>();

  public ApplicationCoverageReport() {

  }

  public ApplicationCoverageReport(List<MuleResource> resources) {
    checkNotNull(resources, "The resources list must not be null.");
    this.resources = resources;
    this.setResourcesWeight();
  }

  public List<MuleResource> getResources() {
    return resources;
  }

  public Double getCoverage() {
    Integer totalMpCount = getApplicationMessageProcessorCount();
    Integer coveredMpCount = getApplicationCoveredMessageProcessorCount();

    if (totalMpCount > 0) {
      return ((double) coveredMpCount * 100) / totalMpCount;
    } else {
      return -1.00;
    }
  }

  public void setBranchCoverage(Double branchCoverage) {
    this.branchCoverage = branchCoverage;
  }

  public Double getBranchCoverage() {
    if (branchCoverage == null) {
      return -1.0;
    }
    return branchCoverage;
  }

  public Integer getApplicationFlowCount() {
    Integer applicationFlowCount = 0;

    for (MuleResource mr : resources) {
      applicationFlowCount += mr.getFlows().size();
    }
    return applicationFlowCount;
  }

  public Integer getApplicationMessageProcessorCount() {
    Integer applicationMpCount = 0;

    for (MuleResource mr : resources) {
      for (MuleFlow mf : mr.getFlows()) {
        applicationMpCount += mf.getLocations().size();
      }
    }
    return applicationMpCount;
  }

  public Integer getApplicationCoveredMessageProcessorCount() {
    Integer applicationMpCount = 0;

    for (MuleResource mr : resources) {
      for (MuleFlow mf : mr.getFlows()) {
        applicationMpCount += mf.getCoveredLocations().size();
      }
    }
    return applicationMpCount;
  }

  private void setResourcesWeight() {
    Integer totalMpCount = getApplicationMessageProcessorCount();
    for (MuleResource mr : resources) {
      Double weight = totalMpCount == 0 ? 0.0 : ((double) mr.getMessageProcessorCount() * 100) / totalMpCount;
      mr.setWeight(weight);
    }
  }
}
