/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.remote;


import static java.util.Collections.emptySet;
import static java.util.Optional.ofNullable;
import static org.apache.commons.lang3.StringUtils.isNotBlank;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.net.URISyntaxException;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.List;
import java.util.Optional;
import java.util.Properties;

import org.apache.commons.cli.ParseException;
import org.mule.munit.common.properties.MUnitUserPropertiesManager;
import org.mule.munit.common.protocol.listeners.RemoteRunEventListener;
import org.mule.munit.common.protocol.listeners.RunEventListener;
import org.mule.munit.common.protocol.listeners.RunEventListenerContainer;
import org.mule.munit.common.util.IllegalPortDefinitionException;
import org.mule.munit.common.util.RunnerPortProvider;
import org.mule.munit.common.util.StackTraceUtil;
import org.mule.munit.plugins.coverage.report.model.ApplicationCoverageReport;
import org.mule.munit.remote.api.client.RunnerClient;
import org.mule.munit.remote.api.configuration.CoverageConfiguration;
import org.mule.munit.remote.api.configuration.NotifierConfiguration;
import org.mule.munit.remote.api.configuration.RunConfiguration;
import org.mule.munit.remote.api.configuration.RunConfigurationParser;
import org.mule.munit.remote.classloading.ClassLoaderUtils;
import org.mule.munit.remote.container.ContainerFactory;
import org.mule.munit.remote.coverage.CoverageManager;
import org.mule.munit.remote.exception.DeploymentException;
import org.mule.munit.remote.exception.DeploymentExceptionThrower;
import org.mule.munit.remote.notifier.NotifierReflectionFactory;
import org.mule.runtime.module.embedded.api.ArtifactConfiguration;
import org.mule.runtime.module.embedded.api.DeploymentConfiguration;
import org.mule.runtime.module.embedded.api.DeploymentService;
import org.mule.runtime.module.embedded.api.EmbeddedContainer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.gson.Gson;

/**
 * This is the main entry point to run a set of MUnit Test Suite files.
 *
 *
 * If could be run from any command line on its own VM provided it receives the proper parameters defined in @link
 * {@link RunConfigurationParser}.
 *
 * The final goal of the class is to run each MUnit Test Suite file provided and report back the run results and the coverage
 * results.
 *
 * @author Mulesoft Inc.
 * @since 2.0.0
 */
public class RemoteRunner {

  private static final boolean ENABLE_XML_VALIDATIONS = true;
  private static final boolean ENABLE_LAZY_INITIALIZATION = true;
  private static final boolean ENABLE_LAZY_CONNECTIONS = true;

  protected transient Logger logger = LoggerFactory.getLogger(this.getClass());

  public static final String SYSTEM_PROPERTIES_FILE = "munit.system.properties.file";
  public static final String MUNIT_DEBUG_LOG_CLASSPATH = "munit.debug.log.classpath";
  public static final String MUNIT_DISABLE_LAZY_CONNECTIONS = "munit.disable.lazy.connections";
  public static final String MUNIT_DISABLE_LAZY_INITIALIZATION = "munit.disable.lazy.initialization";


  private ContainerFactory containerFactory;

  public static void main(String args[]) throws ParseException, IOException, URISyntaxException {
    loadSystemPropertiesFileIfPresent();
    RunConfiguration runConfig = new RunConfigurationParser().parse(args);

    RemoteRunner runner = new RemoteRunner();
    runner.setContainerFactory(new ContainerFactory());
    runner.run(runConfig);
    shutDown();
  }

  public void setContainerFactory(ContainerFactory containerFactory) {
    this.containerFactory = containerFactory;
  }

  public void run(RunConfiguration runConfig) throws IOException, URISyntaxException {
    logger.info("Run Started");
    logClassPathIfNecessary();

    RemoteRunEventListener listener = buildRunnerListener(runConfig.getNotifierConfigurations());
    listener.notifyRunStart();

    Integer munitRunnerPort;
    try {
      munitRunnerPort = new RunnerPortProvider().getPort();
    } catch (IllegalPortDefinitionException e) {
      listener.notifyUnexpectedError(StackTraceUtil.getStackTrace(e));
      listener.notifyRunFinish();
      logger.info("Done");
      return;
    }

    EmbeddedContainer container = containerFactory.createContainer(runConfig.getContainerConfiguration());

    DeploymentConfiguration deploymentConfiguration = buildDeploymentConfiguration();

    CoverageManager coverageManager = buildCoverageManager(runConfig.getCoverageConfiguration());
    try {
      container.start();
      coverageManager.startCoverageServer();

      deployDomain(runConfig.getDomainLocation(), deploymentConfiguration, container);

      deployApplication(runConfig.getProjectName(), deploymentConfiguration,
                        runConfig.getContainerConfiguration().getMunitWorkingDirectoryPath(), container);

      for (String suitePath : runConfig.getSuitePaths()) {
        boolean success = runSuite(runConfig, listener, suitePath, munitRunnerPort);
        if (!success && runConfig.isSkipAfterFailure()) {
          break;
        }
      }

      undeployApplication(container, runConfig.getProjectName());

    } catch (DeploymentException e) {
      listener.notifyContainerFailure(StackTraceUtil.getStackTrace(e));
    } catch (Throwable e) {
      e.printStackTrace();
      listener.notifyUnexpectedError(StackTraceUtil.getStackTrace(e));
    } finally {
      container.stop();
    }

    coverageManager.stopCoverageServer();

    sendCoverageReport(coverageManager, listener);

    listener.notifyRunFinish();
    logger.info("Done");
  }

  protected DeploymentConfiguration buildDeploymentConfiguration() {
    return DeploymentConfiguration.builder()
        .lazyConnectionsEnabled(enableLazyConnections())
        .lazyInitialization(enableLazyInitialization())
        .xmlValidations(ENABLE_XML_VALIDATIONS).build();
  }

  protected void deployDomain(String domainLocation, DeploymentConfiguration deploymentConfiguration, EmbeddedContainer container)
      throws DeploymentException {
    if (isNotBlank(domainLocation)) {
      Path artifactLocation = Paths.get(domainLocation);
      ArtifactConfiguration artifactConfiguration = getArtifactConfiguration(artifactLocation, deploymentConfiguration);
      DeploymentService deploymentService = container.getDeploymentService();
      try {
        deploymentService.deployDomain(artifactConfiguration);
      } catch (RuntimeException e) {
        DeploymentExceptionThrower.throwIfMatches(e);
      }
    }
  }

  protected void deployApplication(String applicationName, DeploymentConfiguration deploymentConfiguration,
                                   String munitWorkingDirectoryPath, EmbeddedContainer container)
      throws DeploymentException {
    Path artifactLocation = Paths.get(munitWorkingDirectoryPath, applicationName);
    ArtifactConfiguration artifactConfiguration = getArtifactConfiguration(artifactLocation, deploymentConfiguration);
    DeploymentService deploymentService = container.getDeploymentService();

    try {
      deploymentService.deployApplication(artifactConfiguration);
    } catch (RuntimeException e) {
      DeploymentExceptionThrower.throwIfMatches(e);
    }
  }

  protected boolean runSuite(RunConfiguration runConfig, RemoteRunEventListener listener, String suitePath, int port) {
    try {
      RunnerClient runnerClient = buildRunnerClient(listener, port);
      runnerClient.sendSuiteRunInfo(runConfig.getRunToken(), suitePath, runConfig.getTestNames(), runConfig.getTags());
      return runnerClient.receiveAndNotify();
    } catch (RuntimeException | IOException | ClassNotFoundException e) {
      listener.notifyContainerFailure(StackTraceUtil.getStackTrace(e));
      return false;
    }
  }

  private void undeployApplication(EmbeddedContainer container, String applicationName) {
    container.getDeploymentService().undeployApplication(applicationName);
  }

  private ArtifactConfiguration getArtifactConfiguration(Path artifactLocation, DeploymentConfiguration deploymentConfiguration) {
    return ArtifactConfiguration.builder()
        .deploymentConfiguration(deploymentConfiguration)
        .artifactLocation(artifactLocation.toFile())
        .build();
  }

  protected RunnerClient buildRunnerClient(RemoteRunEventListener listener, int port) throws IOException {
    return new RunnerClient(port, listener);
  }

  protected RemoteRunEventListener buildRunnerListener(List<NotifierConfiguration> configurations) throws IOException {
    List<RunEventListener> listeners = new NotifierReflectionFactory().createNotifiers(configurations);
    RunEventListenerContainer container = new RunEventListenerContainer();
    listeners.forEach(container::addNotificationListener);
    return container;
  }

  private void logClassPathIfNecessary() {
    if (Boolean.valueOf(System.getProperty(MUNIT_DEBUG_LOG_CLASSPATH, "false"))) {
      logger.info("logging classpath ...");
      new ClassLoaderUtils().getClassPath().forEach(System.out::println);
      logger.info("logging classpath DONE");
    }
  }

  private CoverageManager buildCoverageManager(CoverageConfiguration coverageConfiguration) {
    Integer coverageServerPort = coverageConfiguration.getCoveragePort();
    Boolean randomizeCoveragePort = coverageConfiguration.isRandomizeCoveragePort();

    CoverageManager coverageManager =
        new CoverageManager(randomizeCoveragePort, coverageServerPort, coverageConfiguration.isRunCoverage(),
                            coverageConfiguration.getSuitePaths());
    coverageManager.setIgnoreFlows(ofNullable(coverageConfiguration.getIgnoredFlowNames()).orElse(emptySet()));
    coverageManager.setIgnoreFiles(ofNullable(coverageConfiguration.getIgnoredFiles()).orElse(emptySet()));
    return coverageManager;
  }

  private void sendCoverageReport(CoverageManager coverageManager, RemoteRunEventListener listener) {
    Optional<ApplicationCoverageReport> coverageReport = coverageManager.generateCoverageReport();
    if (coverageReport.isPresent()) {
      String coverageReportJson = new Gson().toJson(coverageReport.get());
      listener.notifyCoverageReport(coverageReportJson);
    }
  }

  private Boolean enableLazyConnections() {
    if (Boolean.valueOf(System.getProperty(MUNIT_DISABLE_LAZY_CONNECTIONS, "false"))) {
      return false;
    }
    return ENABLE_LAZY_CONNECTIONS;
  }

  private Boolean enableLazyInitialization() {
    if (Boolean.valueOf(System.getProperty(MUNIT_DISABLE_LAZY_INITIALIZATION, "false"))) {
      return false;
    }
    return ENABLE_LAZY_INITIALIZATION;
  }

  private static void loadSystemPropertiesFileIfPresent() {
    String filePath = System.getProperty(SYSTEM_PROPERTIES_FILE);
    try {
      if (filePath != null) {
        Properties properties = new Properties();
        properties.load(new FileInputStream(new File(filePath)));
        MUnitUserPropertiesManager.loadPropertiesToSystem(properties);
      }
    } catch (IOException e) {
      // TODO handle this properly
    }
  }

  private static void shutDown() {
    System.exit(0);
  }
}
