/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.remote.exception;

import static com.google.common.base.Preconditions.checkNotNull;

import java.util.List;
import java.util.stream.Collectors;

import com.google.common.base.Throwables;

/**
 * The goal of this class is to find {@link DeploymentException} in a exception cause
 *
 * @author Mulesoft Inc.
 * @since 2.0.0
 */
public class DeploymentExceptionThrower {

  private static final String DEPLOYMENT_EXCEPTION_CANONICAL_NAME = "org.mule.runtime.deployment.model.api.DeploymentException";

  /**
   * It will throw a {@link DeploymentException} if a Mule Deployment exception is found in the causal chain. If not the original
   * exception will be thrown.
   *
   * @param e exception to analyze
   * @throws DeploymentException if a Mule Deployment exception is found
   * @throws Throwable           the original exception
   */
  public static void throwIfMatches(RuntimeException e) throws DeploymentException, RuntimeException {
    checkNotNull("The exception must no be null", e);

    List<Throwable> deploymentExceptions = findCauses(e);
    if (!deploymentExceptions.isEmpty()) {
      throw new DeploymentException(deploymentExceptions.get(0));
    }
    throw e;
  }

  private static List<Throwable> findCauses(Throwable e) {
    List<Throwable> causalChain = Throwables.getCausalChain(e);
    return causalChain.stream()
        .filter(t -> DEPLOYMENT_EXCEPTION_CANONICAL_NAME.equals(t.getClass().getCanonicalName()))
        .collect(Collectors.toList());
  }

}
