/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.remote.api.project;

import static com.google.common.base.Preconditions.checkArgument;
import static org.mule.munit.remote.FolderNames.META_INF;
import static org.mule.munit.remote.FolderNames.MULE_ARTIFACT;

import java.io.IOException;
import java.nio.file.Path;

import org.mule.tools.api.packager.archiver.MuleExplodedArchiver;
import org.mule.tools.api.packager.builder.MulePackageBuilder;
import org.mule.tools.api.packager.packaging.PackagingOptions;
import org.mule.tools.api.packager.sources.DefaultValuesMuleArtifactJsonGenerator;
import org.mule.tools.api.packager.sources.MuleArtifactContentResolver;
import org.mule.tools.api.packager.structure.ProjectStructure;

/**
 * It create a exploded application structure (lightweight) based on a source folder containing all the required folders and
 * files. The exploded application contains the test resources.
 *
 * @author Mulesoft Inc.
 * @since 2.0.0
 */
public class MuleApplicationStructureGenerator implements ApplicationStructureGenerator {

  public static final boolean ONLY_MULE_SOURCES = false;
  public static final boolean LIGHTWEIGHT_PACKAGE = true;
  public static final boolean ATTACH_MULE_SOURCES = false;
  public static final boolean TEST_PACKAGE = true;

  private Path baseFolderPath;
  private Path originFolderPath;
  private MulePackageBuilder packageBuilder;


  public MuleApplicationStructureGenerator(Path baseFolderPath, Path sourceFolder) {
    checkArgument(baseFolderPath != null, "The base folder must not be null");
    checkArgument(sourceFolder != null, "The source folder must not be null");

    this.baseFolderPath = baseFolderPath;
    this.originFolderPath = sourceFolder;
    this.packageBuilder = new MulePackageBuilder();
  }

  public MulePackageBuilder getPackageBuilder() {
    return packageBuilder;
  }

  public void setPackageBuilder(MulePackageBuilder packageBuilder) {
    this.packageBuilder = packageBuilder;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Path generate(Path destinationFolder) throws Exception {
    checkArgument(destinationFolder != null, "The destination folder must not be null");
    createExplodedPackage(destinationFolder);
    reGenerateMuleArtifactJson(destinationFolder);

    return destinationFolder;
  }

  private void createExplodedPackage(Path destinationFolderPath) throws Exception {
    packageBuilder
        .withPackagingOptions(new PackagingOptions(ONLY_MULE_SOURCES, LIGHTWEIGHT_PACKAGE, ATTACH_MULE_SOURCES, TEST_PACKAGE))
        .withArchiver(new MuleExplodedArchiver())
        .createPackage(originFolderPath, destinationFolderPath);
  }

  private void reGenerateMuleArtifactJson(Path destinationFolderPath) throws IOException {
    Path muleArtifactFolderPath = destinationFolderPath.resolve(META_INF.value()).resolve(MULE_ARTIFACT.value());

    ProjectStructure projectStructure = new ProjectStructure(baseFolderPath, TEST_PACKAGE);
    MuleArtifactContentResolver muleArtifactContentResolver = new MuleArtifactContentResolver(projectStructure);


    DefaultValuesMuleArtifactJsonGenerator.generate(muleArtifactFolderPath, muleArtifactFolderPath,
                                                    muleArtifactContentResolver);

  }

}
