/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.remote.container;

import static org.apache.commons.lang3.StringUtils.isNotBlank;

import java.nio.file.Path;
import java.nio.file.Paths;

import org.mule.munit.remote.api.configuration.RunConfiguration;
import org.mule.munit.remote.exception.DeploymentException;
import org.mule.munit.remote.exception.DeploymentExceptionThrower;
import org.mule.munit.remote.properties.TemporaryFolderConfigurator;
import org.mule.runtime.module.embedded.api.ArtifactConfiguration;
import org.mule.runtime.module.embedded.api.DeploymentConfiguration;
import org.mule.runtime.module.embedded.api.DeploymentService;
import org.mule.runtime.module.embedded.api.EmbeddedContainer;


/**
 * Wrapper for the {@link EmbeddedContainer} that provides a friendlier API
 *
 * @author Mulesoft Inc.
 * @since 2.2.0
 */
public class ContainerManager {


  private EmbeddedContainer container;
  private RunConfiguration runConfiguration;
  private DeploymentConfiguration deploymentConfiguration;

  public ContainerManager(EmbeddedContainer container, RunConfiguration runConfiguration,
                          DeploymentConfiguration deploymentConfiguration) {
    this.container = container;
    this.runConfiguration = runConfiguration;
    this.deploymentConfiguration = deploymentConfiguration;
  }

  public void deployDomain() throws DeploymentException {
    String domainLocation = runConfiguration.getDomainLocation();
    if (isNotBlank(domainLocation)) {
      Path artifactLocation = Paths.get(domainLocation);
      ArtifactConfiguration artifactConfiguration = getArtifactConfiguration(artifactLocation, deploymentConfiguration);
      DeploymentService deploymentService = container.getDeploymentService();
      try {
        deploymentService.deployDomain(artifactConfiguration);
      } catch (RuntimeException e) {
        DeploymentExceptionThrower.throwIfMatches(e);
      }
    }
  }

  public void deployApplication() throws DeploymentException {
    String applicationName = runConfiguration.getProjectName();
    String munitWorkingDirectoryPath = runConfiguration.getContainerConfiguration().getMunitWorkingDirectoryPath();
    Path artifactLocation = Paths.get(munitWorkingDirectoryPath, applicationName);
    ArtifactConfiguration artifactConfiguration = getArtifactConfiguration(artifactLocation, deploymentConfiguration);
    DeploymentService deploymentService = container.getDeploymentService();
    TemporaryFolderConfigurator.loadTemporaryDirectoryProperty();
    try {
      deploymentService.deployApplication(artifactConfiguration);
    } catch (RuntimeException e) {
      DeploymentExceptionThrower.throwIfMatches(e);
    }
  }

  public void undeployApplication() {
    container.getDeploymentService().undeployApplication(runConfiguration.getProjectName());
  }

  private ArtifactConfiguration getArtifactConfiguration(Path artifactLocation, DeploymentConfiguration deploymentConfiguration) {
    return ArtifactConfiguration.builder()
        .deploymentConfiguration(deploymentConfiguration)
        .artifactLocation(artifactLocation.toFile())
        .build();
  }

}
