/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.remote.container;

import static java.util.Collections.disjoint;
import static org.apache.commons.lang3.StringUtils.EMPTY;

import java.io.File;
import java.io.IOException;
import java.util.Collection;
import java.util.Set;

import org.mule.munit.common.properties.MUnitUserPropertiesManager;
import org.mule.munit.common.protocol.listeners.RemoteRunEventListener;
import org.mule.munit.common.util.StackTraceUtil;
import org.mule.munit.remote.api.client.RunnerClient;
import org.mule.munit.remote.api.configuration.RunConfiguration;
import org.mule.munit.remote.exception.DeploymentException;
import org.mule.munit.remote.properties.ParameterizedSuite;
import org.mule.munit.remote.properties.ParameterizedSuitesLocator;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Dispatches a Suite Run to the {@link ContainerManager} with a given {@link RunConfiguration}
 *
 * @author Mulesoft Inc.
 * @since 2.2.0
 */
public class SuiteRunDispatcher {

  private static final transient Logger LOGGER = LoggerFactory.getLogger(SuiteRunDispatcher.class);

  private ContainerManager containerManager;
  private RunConfiguration runConfig;
  private Integer munitRunnerPort;
  private ParameterizedSuitesLocator parameterizedSuitesLocator;

  public SuiteRunDispatcher(ContainerManager containerManager, Integer munitRunnerPort, RunConfiguration runConfiguration) {
    this.containerManager = containerManager;
    this.runConfig = runConfiguration;
    this.munitRunnerPort = munitRunnerPort;
    File appLocation =
        new File(runConfiguration.getContainerConfiguration().getMunitWorkingDirectoryPath(), runConfiguration.getProjectName());
    this.parameterizedSuitesLocator = new ParameterizedSuitesLocator(appLocation, runConfig.getAllSuitePaths());
  }

  public void runSuites(RemoteRunEventListener listener) throws DeploymentException {
    try {
      runParameterizedSuites(parameterizedSuitesLocator.parameterizedSuites(), listener);
      runNonParameterizedSuites(parameterizedSuitesLocator.nonParameterizedSuites(), listener);
    } catch (SkipAfterFailureException e) {
      LOGGER.debug("Skipped running suites since skipAfterFailure is on");
    }
  }

  private void runParameterizedSuites(Collection<ParameterizedSuite> parameterizedSuites, RemoteRunEventListener listener)
      throws SkipAfterFailureException {
    setInitialProperties(parameterizedSuites);
    for (ParameterizedSuite suite : parameterizedSuites) {
      if (shouldRunSuite(suite.getSuitePath())) {
        MUnitUserPropertiesManager.addUserPropertiesToSystem(suite.getParameters());
        try {
          containerManager.deployApplication();
          boolean success = runSuite(runConfig, listener, suite.getSuitePath(), suite.getParameterizationName());
          if (!success && runConfig.isSkipAfterFailure()) {
            throw new SkipAfterFailureException();
          }
        } catch (DeploymentException e) {
          listener.notifyContainerFailure(suite.getSuitePath(), suite.getParameterizationName(), StackTraceUtil.getStackTrace(e));
        } finally {
          containerManager.undeployApplication();
        }
      }
    }
  }

  private void runNonParameterizedSuites(Set<String> suitePaths, RemoteRunEventListener listener)
      throws DeploymentException, SkipAfterFailureException {
    if (suitePaths.isEmpty() || disjoint(suitePaths, runConfig.getSuitePaths())) {
      return;
    }
    containerManager.deployApplication();
    try {
      for (String suitePath : suitePaths) {
        if (shouldRunSuite(suitePath)) {
          boolean success = runSuite(runConfig, listener, suitePath, EMPTY);
          if (!success && runConfig.isSkipAfterFailure()) {
            throw new SkipAfterFailureException();
          }
        }
      }
    } finally {
      containerManager.undeployApplication();
    }
  }

  private boolean runSuite(RunConfiguration runConfig, RemoteRunEventListener listener, String suitePath,
                           String parameterizationName) {
    try {
      RunnerClient runnerClient = getRunnerClient(listener);
      runnerClient.sendSuiteRunInfo(runConfig.getRunToken(), suitePath, parameterizationName, runConfig.getTestNames(),
                                    runConfig.getTags());
      return runnerClient.receiveAndNotify();
    } catch (RuntimeException | IOException | ClassNotFoundException e) {
      listener.notifyContainerFailure(suitePath, parameterizationName, StackTraceUtil.getStackTrace(e));
      return false;
    }
  }

  protected RunnerClient getRunnerClient(RemoteRunEventListener listener) throws IOException {
    return new RunnerClient(munitRunnerPort, listener);
  }



  private boolean shouldRunSuite(String suitePath) {
    return runConfig.getSuitePaths().contains(suitePath);
  }

  private class SkipAfterFailureException extends Exception {

  }

  /**
   * This step is to ensure that the deployment is successful whether parameterized suites are run or not
   */
  private void setInitialProperties(Collection<ParameterizedSuite> parameterizedSuites) {
    parameterizedSuites.forEach(suite -> MUnitUserPropertiesManager.addUserPropertiesToSystem(suite.getParameters()));
  }

}
