/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.remote.properties;

import static java.util.Collections.emptySet;

import java.io.File;
import java.io.IOException;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.jdom2.Document;
import org.jdom2.Element;
import org.jdom2.JDOMException;
import org.jdom2.Namespace;
import org.jdom2.input.SAXBuilder;

/**
 * Parses the parameterization configuration and returns the parameterized suites, if present
 *
 * @author Mulesoft Inc.
 * @since 2.2.0
 */
public class ParameterizedSuiteParser {

  // Schema names and namespaces
  private static final Namespace MUNIT_NAMESPACE = Namespace.getNamespace("munit", "http://www.mulesoft.org/schema/mule/munit");
  private static final String CONFIG_NAME = "config";
  private static final String PARAMETERIZATIONS_NAME = "parameterizations";
  private static final String PARAMETERIZATION_NAME = "parameterization";
  private static final String PARAMETERS_NAME = "parameters";
  private static final String PARAMETER_NAME = "parameter";
  private static final String PROPERTY_NAME_NAME = "propertyName";
  private static final String VALUE_NAME = "value";
  private static final String NAME_NAME = "name";

  public static Set<ParameterizedSuite> parseParameterizations(File suiteFile, String suitePath) {
    try {
      SAXBuilder saxBuilder = new SAXBuilder();
      Document document = saxBuilder.build(suiteFile);
      Element rootElement = document.getRootElement();
      Element config = rootElement.getChild(CONFIG_NAME, MUNIT_NAMESPACE);
      if (config == null) {
        return emptySet();
      }
      Element parameterizations = config.getChild(PARAMETERIZATIONS_NAME, MUNIT_NAMESPACE);
      if (parameterizations == null) {
        return emptySet();
      }
      return collectParameterizedSuites(parameterizations.getChildren(PARAMETERIZATION_NAME, MUNIT_NAMESPACE), suitePath);
    } catch (JDOMException | IOException e) {
      // In case of any parsing error, we should assume it is not parameterized, and then let the container validate the xml
      return emptySet();
    }
  }

  private static Set<ParameterizedSuite> collectParameterizedSuites(List<Element> parameterizations, String suitePath) {
    Set<ParameterizedSuite> parameterizedSuites = new HashSet<>();
    for (Element parameterization : parameterizations) {
      Map<String, String> parametersMap = new HashMap<>();
      Element parameters = parameterization.getChild(PARAMETERS_NAME, MUNIT_NAMESPACE);
      if (parameters != null) {
        List<Element> parameterChildren = parameters.getChildren(PARAMETER_NAME, MUNIT_NAMESPACE);
        for (Element parameter : parameterChildren) {
          parametersMap.put(parameter.getAttributeValue(PROPERTY_NAME_NAME), parameter.getAttributeValue(VALUE_NAME));
        }
      }
      String name = parameterization.getAttributeValue(NAME_NAME);
      parameterizedSuites.add(new ParameterizedSuite(suitePath, name, parametersMap));
    }
    return parameterizedSuites;
  }

}
