/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.remote.api.configuration;

import java.util.List;
import java.util.Set;

/**
 * Defines the information required to run MUnit suites
 *
 * @author Mulesoft Inc.
 * @since 2.0.0
 */
public class RunConfiguration {

  private String runToken;
  private String projectName;
  private String domainLocation;

  private Set<String> tags;
  private Set<String> testNames;
  private Set<String> suitePaths;
  private Set<String> allSuitePaths;
  private Boolean skipAfterFailure;

  private List<NotifierConfiguration> notifierConfigurations;
  private CoverageConfiguration coverageConfiguration;
  private ContainerConfiguration containerConfiguration;

  protected RunConfiguration(String runToken, String domainLocation, String projectName, Set<String> tags, Set<String> testNames,
                             Set<String> suitePaths, Set<String> allSuitePaths, Boolean skipAfterFailure,
                             List<NotifierConfiguration> notifierConfigurations,
                             CoverageConfiguration coverageConfiguration, ContainerConfiguration containerConfiguration) {
    this.runToken = runToken;
    this.domainLocation = domainLocation;
    this.projectName = projectName;
    this.tags = tags;
    this.testNames = testNames;
    this.suitePaths = suitePaths;
    this.allSuitePaths = allSuitePaths;
    this.skipAfterFailure = skipAfterFailure;
    this.notifierConfigurations = notifierConfigurations;
    this.coverageConfiguration = coverageConfiguration;
    this.containerConfiguration = containerConfiguration;
  }

  public String getRunToken() {
    return runToken;
  }

  public String getDomainLocation() {
    return domainLocation;
  }

  public String getProjectName() {
    return projectName;
  }

  public Set<String> getTags() {
    return tags;
  }

  public Set<String> getTestNames() {
    return testNames;
  }

  public Set<String> getSuitePaths() {
    return suitePaths;
  }

  public Set<String> getAllSuitePaths() {
    return allSuitePaths;
  }

  public Boolean isSkipAfterFailure() {
    return skipAfterFailure;
  }

  public List<NotifierConfiguration> getNotifierConfigurations() {
    return notifierConfigurations;
  }

  public CoverageConfiguration getCoverageConfiguration() {
    return coverageConfiguration;
  }

  public ContainerConfiguration getContainerConfiguration() {
    return containerConfiguration;
  }

  /**
   * Run Configuration Builder
   *
   * @author Mulesoft Inc.
   * @since 2.0.0
   */
  public static class RunConfigurationBuilder {

    private String runToken;
    private String projectName;
    private String domainLocation;

    private Set<String> tags;
    private Set<String> testNames;
    private Set<String> suitePaths;
    private Set<String> allSuitePaths;
    private Boolean skipAfterFailure;

    private List<NotifierConfiguration> notifierConfigurations;
    private CoverageConfiguration coverageConfiguration;
    private ContainerConfiguration containerConfiguration;

    public static RunConfigurationBuilder from(RunConfiguration rc) {
      RunConfigurationBuilder builder = new RunConfigurationBuilder();

      builder.withRunToken(rc.getRunToken())
          .withDomainLocation(rc.getDomainLocation())
          .withProjectName(rc.getProjectName())
          .withTags(rc.getTags())
          .withTestNames(rc.getTestNames())
          .withSuitePaths(rc.getSuitePaths())
          .withAllSuitePaths(rc.getAllSuitePaths())
          .withSkipAfterFailure(rc.isSkipAfterFailure())
          .withNotifierConfigurations(rc.getNotifierConfigurations())
          .withCoverageConfiguration(rc.getCoverageConfiguration())
          .withContainerConfiguration(rc.getContainerConfiguration());


      return builder;
    }

    public RunConfigurationBuilder withRunToken(String runToken) {
      this.runToken = runToken;
      return this;
    }

    public RunConfigurationBuilder withDomainLocation(String domainLocation) {
      this.domainLocation = domainLocation;
      return this;
    }

    public RunConfigurationBuilder withProjectName(String projectName) {
      this.projectName = projectName;
      return this;
    }

    public RunConfigurationBuilder withTags(Set<String> tags) {
      this.tags = tags;
      return this;
    }

    public RunConfigurationBuilder withTestNames(Set<String> testNames) {
      this.testNames = testNames;
      return this;
    }

    public RunConfigurationBuilder withSuitePaths(Set<String> suitePaths) {
      this.suitePaths = suitePaths;
      return this;
    }

    public RunConfigurationBuilder withAllSuitePaths(Set<String> allSuitePaths) {
      this.allSuitePaths = allSuitePaths;
      return this;
    }

    public RunConfigurationBuilder withSkipAfterFailure(Boolean skipAfterFailure) {
      this.skipAfterFailure = skipAfterFailure;
      return this;
    }

    public RunConfigurationBuilder withNotifierConfigurations(List<NotifierConfiguration> notifierConfigurations) {
      this.notifierConfigurations = notifierConfigurations;
      return this;
    }

    public RunConfigurationBuilder withCoverageConfiguration(CoverageConfiguration coverageConfiguration) {
      this.coverageConfiguration = coverageConfiguration;
      return this;
    }

    public RunConfigurationBuilder withContainerConfiguration(ContainerConfiguration containerConfiguration) {
      this.containerConfiguration = containerConfiguration;
      return this;
    }

    public RunConfiguration build() {
      return new RunConfiguration(runToken, domainLocation, projectName, tags, testNames, suitePaths, allSuitePaths,
                                  skipAfterFailure, notifierConfigurations, coverageConfiguration, containerConfiguration);
    }
  }
}
