/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.remote.container;

import static com.google.common.collect.Sets.newHashSet;
import static java.lang.String.format;
import static java.util.Optional.empty;
import static java.util.Optional.of;
import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toSet;
import static org.apache.commons.lang3.StringUtils.EMPTY;
import static org.mule.munit.common.util.StackTraceUtil.getStackTrace;
import static org.mule.munit.common.util.VersionUtils.isAtLeastMinMuleVersion;
import static org.mule.munit.remote.FolderNames.META_INF;
import static org.mule.munit.remote.FolderNames.MULE_ARTIFACT;
import static org.mule.tools.api.packager.structure.PackagerFiles.MULE_ARTIFACT_JSON;

import java.io.File;
import java.io.IOException;
import java.nio.charset.Charset;
import java.nio.file.Paths;
import java.util.*;
import java.util.stream.Collectors;

import org.mule.munit.common.protocol.listeners.RemoteRunEventListener;
import org.mule.munit.common.util.FileUtils;
import org.mule.munit.common.util.IOUtils;
import org.mule.munit.remote.api.client.RunnerClient;
import org.mule.munit.remote.api.configuration.RunConfiguration;
import org.mule.munit.remote.container.model.SuiteDeployment;
import org.mule.munit.remote.container.model.SuiteRun;
import org.mule.munit.remote.exception.DeploymentException;
import org.mule.munit.remote.properties.deploy.DeploymentProperties;
import org.mule.munit.remote.properties.Parameterization;
import org.mule.munit.remote.properties.deploy.MuleRuntimeDeploymentProperties;
import org.mule.munit.remote.properties.deploy.TemporaryFolderProperties;
import org.mule.runtime.api.deployment.meta.MuleApplicationModel;
import org.mule.runtime.api.deployment.persistence.MuleApplicationModelJsonSerializer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Dispatches a Suite Run to the {@link ContainerManager} with a given {@link RunConfiguration}
 *
 * @author Mulesoft Inc.
 * @since 2.2.0
 */
public class SuiteRunDispatcher {

  private static final transient Logger LOGGER = LoggerFactory.getLogger(SuiteRunDispatcher.class);

  private final ContainerManager containerManager;
  private final RunConfiguration runConfig;
  private final Integer munitRunnerPort;
  private final MuleApplicationModel originalMuleApplicationModel;
  private final Set<SuiteRun> suiteRuns;
  private final Collection<DeploymentProperties> deploymentProperties;

  public SuiteRunDispatcher(ContainerManager containerManager, Integer munitRunnerPort, RunConfiguration runConfiguration,
                            Set<SuiteRun> suiteRuns) {
    this.containerManager = containerManager;
    this.runConfig = runConfiguration;
    this.munitRunnerPort = munitRunnerPort;
    this.suiteRuns = suiteRuns;
    this.originalMuleApplicationModel = getOriginalApplicationModel();
    this.deploymentProperties = getDeploymentProperties();
  }

  private MuleApplicationModel getOriginalApplicationModel() {
    try {
      File muleArtifactJsonFile = getMuleArtifactJsonFile();
      return new MuleApplicationModelJsonSerializer()
          .deserialize(IOUtils.toString(muleArtifactJsonFile.toURI(), Charset.defaultCharset()));
    } catch (Exception e) {
      throw new IllegalStateException(e);
    }
  }


  private Collection<DeploymentProperties> getDeploymentProperties() {
    List<DeploymentProperties> deploymentProperties = new ArrayList<>();
    deploymentProperties.add(new MuleRuntimeDeploymentProperties(runConfig));
    deploymentProperties.add(new TemporaryFolderProperties());
    return deploymentProperties;
  }

  private File getMuleArtifactJsonFile() {
    File applicationDirectory =
        new File(runConfig.getContainerConfiguration().getMunitWorkingDirectoryPath(), runConfig.getProjectName());
    return Paths.get(applicationDirectory.toURI()).resolve(META_INF.value()).resolve(MULE_ARTIFACT.value())
        .resolve(MULE_ARTIFACT_JSON).toFile();
  }

  public void runSuites(RemoteRunEventListener listener) throws DeploymentException {
    try {
      Collection<SuiteDeployment> suiteDeployments = generateSuiteDeployments();
      Map<String, String> initialProperties = getInitialProperties(suiteDeployments);
      for (SuiteDeployment suiteDeployment : suiteDeployments) {
        performSuiteDeploy(listener, suiteDeployment, initialProperties);
      }
    } catch (SkipAfterFailureException e) {
      LOGGER.debug("Skipped running suites since skipAfterFailure is on");
    }
  }

  private Collection<SuiteDeployment> generateSuiteDeployments() {
    List<SuiteDeployment> suiteDeployments = new ArrayList<>();
    suiteDeployments.addAll(getParameterizedDeployments());
    getDesignTimeSuitesDeployment().ifPresent(suiteDeployments::add);
    getRuntimeSuitesDeployment().ifPresent(suiteDeployments::add);
    return suiteDeployments;
  }

  private void performSuiteDeploy(RemoteRunEventListener listener, SuiteDeployment suiteDeployment,
                                  Map<String, String> initialProperties)
      throws SkipAfterFailureException {
    Set<SuiteRun> suiteRuns = suiteDeployment.getSuiteRuns();
    if (suiteRuns.stream().noneMatch(this::shouldRunSuite)) {
      LOGGER.info("Suite run for suites " + suiteRuns + " will not start since no suites will run");
      return;
    }
    try {
      removeXmlInvalidSuites(getXmlInvalidSuites(suiteDeployment));
      Map<String, String> systemProperties = getSystemProperties(initialProperties, suiteDeployment);
      containerManager.deployApplication(suiteDeployment.isEnableXmlValidations(), systemProperties);
      for (SuiteRun suiteRun : suiteRuns) {
        if (shouldRunSuite(suiteRun)) {
          String suitePath = suiteRun.getSuitePath();
          String parameterizationName = getParameterizationName(suiteRun);
          boolean success = runSuite(runConfig, listener, suitePath, parameterizationName);
          if (!success && runConfig.isSkipAfterFailure()) {
            throw new SkipAfterFailureException();
          }
        } else {
          showSuiteSkippedMessage(suiteRun);
        }
      }
    } catch (DeploymentException e) {
      if (suiteRuns.size() > 1) {
        listener.notifyContainerFailure(getStackTrace(e));
      } else {
        SuiteRun suiteRun = suiteRuns.iterator().next();
        listener.notifyContainerFailure(suiteRun.getSuitePath(), getParameterizationName(suiteRun), getStackTrace(e));
      }
    } finally {
      containerManager.undeployApplication();
    }
  }

  private Set<String> getXmlInvalidSuites(SuiteDeployment suiteDeployment) {
    Set<String> xmlInvalidSuites = new HashSet<>();
    if (suiteDeployment.isEnableXmlValidations()) {
      xmlInvalidSuites.addAll(getDesignTimeSuitesPaths());
    }
    xmlInvalidSuites.addAll(suiteDeployment.getSuiteRuns().stream().filter(suiteRun -> !shouldRunSuite(suiteRun))
        .map(SuiteRun::getSuitePath).collect(Collectors.toList()));
    return xmlInvalidSuites;
  }

  /**
   * Suites can have invalid or incomplete xml, we need to exclude them during a deploy where xml validations are on
   */
  private void removeXmlInvalidSuites(Set<String> invalidXmlSuites) {
    Set<String> newConfigs = originalMuleApplicationModel.getConfigs();
    if (!invalidXmlSuites.isEmpty()) {
      newConfigs = newConfigs.stream().filter(config -> !invalidXmlSuites.contains(config)).collect(toSet());
    }
    try {
      MuleApplicationModel muleApplicationModel = overrideConfigs(originalMuleApplicationModel, newConfigs);
      String newMuleApplicationModelString = new MuleApplicationModelJsonSerializer().serialize(muleApplicationModel);
      FileUtils.write(getMuleArtifactJsonFile(), newMuleApplicationModelString, Charset.defaultCharset());
    } catch (IOException e) {
      throw new IllegalStateException("An error occurred while regenerating the Mule Application Model", e);
    }
  }

  private Map<String, String> getSystemProperties(Map<String, String> initialProperties, SuiteDeployment suiteDeployment) {
    Map<String, String> systemProperties = new HashMap<>(initialProperties);
    deploymentProperties.forEach(properties -> systemProperties.putAll(properties.get()));
    systemProperties.putAll(suiteDeployment.getSystemProperties());
    return systemProperties;
  }

  private void showSuiteSkippedMessage(SuiteRun suiteRun) {
    String message = "Suite " + suiteRun.getSuitePath() + " will not be deployed: ";
    if (suiteRun.isIgnored()) {
      message += "Suite is ignored";
    } else if (!isCurrentRuntimeAtLeastMinVersion(suiteRun)) {
      message += format("Current runtime version [%s] is lower that suite's minMuleVersion [%s]", getRuntimeVersion(),
                        suiteRun.getMinMuleVersion().orElse("N/A"));
    } else if (!isSuiteFiltered(suiteRun)) {
      message += "Suite was filtered from running";
    }
    LOGGER.info(message);
  }

  private boolean runSuite(RunConfiguration runConfig, RemoteRunEventListener listener, String suitePath,
                           String parameterizationName) {
    try {
      RunnerClient runnerClient = getRunnerClient(listener);
      runnerClient.sendSuiteRunInfo(runConfig.getRunToken(), suitePath, parameterizationName, runConfig.getTestNames(),
                                    runConfig.getTags());
      return runnerClient.receiveAndNotify();
    } catch (RuntimeException | IOException | ClassNotFoundException e) {
      listener.notifyContainerFailure(suitePath, parameterizationName, getStackTrace(e));
      return false;
    }
  }

  protected RunnerClient getRunnerClient(RemoteRunEventListener listener) throws IOException {
    return new RunnerClient(munitRunnerPort, listener);
  }

  private boolean shouldRunSuite(SuiteRun suiteRun) {
    return !suiteRun.isIgnored() && isCurrentRuntimeAtLeastMinVersion(suiteRun) && isSuiteFiltered(suiteRun);
  }

  private Boolean isCurrentRuntimeAtLeastMinVersion(SuiteRun suiteRun) {
    return suiteRun.getMinMuleVersion().map(minMuleVersion -> isAtLeastMinMuleVersion(getRuntimeVersion(), minMuleVersion))
        .orElse(true);
  }

  private boolean isSuiteFiltered(SuiteRun suiteRun) {
    return runConfig.getSuitePaths().contains(suiteRun.getSuitePath());
  }

  private class SkipAfterFailureException extends Exception {

  }

  /**
   * This step is to ensure that the deployment is successful whether parameterized suites are run or not
   */
  private Map<String, String> getInitialProperties(Collection<SuiteDeployment> suiteDeployments) {
    Map<String, String> initialProperties = new HashMap<>();
    suiteDeployments.forEach(suite -> initialProperties.putAll(suite.getSystemProperties()));
    return initialProperties;
  }

  public MuleApplicationModel overrideConfigs(MuleApplicationModel originalMuleApplicationModel, Set<String> configs) {
    MuleApplicationModel.MuleApplicationModelBuilder builder = new MuleApplicationModel.MuleApplicationModelBuilder();
    builder.setConfigs(configs);
    builder.setMinMuleVersion(originalMuleApplicationModel.getMinMuleVersion());
    builder.setName(originalMuleApplicationModel.getName());
    builder.setRedeploymentEnabled(true);
    builder.setRequiredProduct(originalMuleApplicationModel.getRequiredProduct());
    builder.setSecureProperties(originalMuleApplicationModel.getSecureProperties());
    builder.withBundleDescriptorLoader(originalMuleApplicationModel.getBundleDescriptorLoader());
    builder.withClassLoaderModelDescriptorLoader(originalMuleApplicationModel.getClassLoaderModelLoaderDescriptor());
    return builder.build();
  }

  private List<SuiteDeployment> getParameterizedDeployments() {
    return suiteRuns.stream().filter(suiteRun -> suiteRun.getParameterization().isPresent())
        .map(this::createParameterizedSuiteDeployment).collect(toList());
  }

  private SuiteDeployment createParameterizedSuiteDeployment(SuiteRun suiteRun) {
    Parameterization parameterization = suiteRun.getParameterization().get();
    return SuiteDeployment.builder().withSuiteRuns(newHashSet(suiteRun)).withSystemProperties(parameterization.getParameters())
        .withEnableXmlValidations(!suiteRun.isDesignTime()).build();
  }

  private Set<SuiteRun> getDesignTimeSuites() {
    return suiteRuns.stream().filter(SuiteRun::isDesignTime).collect(Collectors.toSet());
  }

  private Optional<SuiteDeployment> getDesignTimeSuitesDeployment() {
    Set<SuiteRun> designTimeSuites =
        getDesignTimeSuites().stream().filter(suiteRun -> !suiteRun.getParameterization().isPresent()).collect(toSet());
    if (designTimeSuites.isEmpty()) {
      return Optional.empty();
    }
    return of(SuiteDeployment.builder().withSuiteRuns(designTimeSuites).withEnableXmlValidations(false).build());
  }

  private Optional<SuiteDeployment> getRuntimeSuitesDeployment() {
    Set<SuiteRun> runtimeSuites = suiteRuns.stream().filter(suiteRun -> !suiteRun.isDesignTime())
        .filter(suiteRun -> !suiteRun.getParameterization().isPresent()).collect(toSet());
    if (runtimeSuites.isEmpty()) {
      return empty();
    }
    return of(SuiteDeployment.builder().withSuiteRuns(runtimeSuites).withEnableXmlValidations(true).build());
  }

  private Set<String> getDesignTimeSuitesPaths() {
    return getDesignTimeSuites().stream().map(SuiteRun::getSuitePath).collect(toSet());
  }

  private String getParameterizationName(SuiteRun suiteRun) {
    return suiteRun.getParameterization().map(Parameterization::getParameterizationName).orElse(EMPTY);
  }

  private String getRuntimeVersion() {
    return runConfig.getContainerConfiguration().getRuntimeId();
  }

}
