/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.remote.container;

import static com.google.common.base.Preconditions.checkArgument;

import java.io.File;
import java.util.HashSet;
import java.util.Set;

import org.mule.munit.remote.api.configuration.RunConfiguration;
import org.mule.munit.remote.container.model.SuiteRun;
import org.mule.munit.remote.properties.DesignTimeSuitesParser;
import org.mule.munit.remote.properties.Parameterization;
import org.mule.munit.remote.properties.SuiteConfigParser;

/**
 * Locates and generates {@link SuiteRun} based on the {@link RunConfiguration} and the suites located in the application. If the
 * suite is parameterized, it generates a {@link SuiteRun} for each {@link Parameterization}.
 * 
 * @author Mulesoft Inc.
 * @since 2.2.0
 */
public class SuiteRunLocator {

  private static final String DESIGN_TIME_MIN_MULE_VERSION = "4.2.0";

  private RunConfiguration runConfiguration;
  private DesignTimeSuitesParser designTimeSuitesParser;

  public SuiteRunLocator(RunConfiguration runConfiguration) {
    checkArgument(runConfiguration != null, "Run configuration cannot be null");
    this.runConfiguration = runConfiguration;
    this.designTimeSuitesParser = new DesignTimeSuitesParser();
  }

  public Set<SuiteRun> locate() {
    File appLocation =
        new File(runConfiguration.getContainerConfiguration().getMunitWorkingDirectoryPath(), runConfiguration.getProjectName());
    Set<SuiteRun> suiteRuns = new HashSet<>();
    for (String suitePath : runConfiguration.getAllSuitePaths()) {
      File suiteFile = new File(appLocation, suitePath);
      SuiteConfigParser suiteConfigParser = new SuiteConfigParser(suiteFile, appLocation);
      Set<Parameterization> parameterizations = suiteConfigParser.parseParameterizations();
      boolean hasDesignTimeTests = designTimeSuitesParser.hasDesignTimeTests(suiteFile);
      boolean isIgnored = suiteConfigParser.isIgnored();
      String minMuleVersion =
          suiteConfigParser.getMinMuleVersion().orElse(hasDesignTimeTests ? DESIGN_TIME_MIN_MULE_VERSION : null);
      if (parameterizations.isEmpty()) {
        suiteRuns.add(SuiteRun.builder().withSuitePath(suitePath).withDesignTime(hasDesignTimeTests)
            .withMinMuleVersion(minMuleVersion).withIgnore(isIgnored).build());
      } else {
        for (Parameterization parameterization : parameterizations) {
          suiteRuns.add(SuiteRun.builder().withSuitePath(suitePath)
              .withParameterization(parameterization)
              .withDesignTime(hasDesignTimeTests)
              .withIgnore(isIgnored)
              .withMinMuleVersion(minMuleVersion)
              .build());
        }
      }
    }
    return suiteRuns;
  }
}
