/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.remote.api.client;

import static org.mule.munit.common.protocol.message.MessageField.MUNIT_SUITE_KEY;
import static org.mule.munit.common.protocol.message.MessageField.PARAMETERIZATION_KEY;
import static org.mule.munit.common.protocol.message.MessageField.RUN_TOKEN_KEY;
import static org.mule.munit.common.protocol.message.MessageField.SEPARATOR_TOKEN;
import static org.mule.munit.common.protocol.message.MessageField.TAGS_KEY;
import static org.mule.munit.common.protocol.message.MessageField.TEST_NAMES_KEY;
import static org.mule.munit.common.protocol.message.MessageID.RUN_SUITE;
import static org.mule.munit.common.protocol.message.TestStatus.ERROR;
import static org.mule.munit.common.protocol.message.TestStatus.FAILURE;

import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.net.ConnectException;
import java.net.Socket;
import java.util.Collection;
import java.util.Set;
import java.util.stream.Collectors;

import org.apache.commons.lang3.StringUtils;

import org.mule.munit.common.protocol.listeners.RemoteRunEventListener;
import org.mule.munit.common.protocol.listeners.RunEventListener;
import org.mule.munit.common.protocol.listeners.RunEventListenerContainer;
import org.mule.munit.common.protocol.message.RunMessage;
import org.mule.munit.common.protocol.message.RunMessageParser;
import org.mule.munit.common.protocol.message.TestStatus;

import com.google.common.collect.ImmutableMap;
import com.google.gson.Gson;

/**
 * Sends messages to the munit-runner plugin.
 *
 * @author Mulesoft Inc.
 * @since 2.0.0
 */
public class RunnerClient {

  public static final Integer RECONNECT_MAX_ATTEMPTS = 3;
  public static final int CONNECTION_SLEEP_MILLIS = 1000;

  private int port;
  private RemoteRunEventListener runnerEventListener;
  private ObjectInputStream in;
  private ObjectOutputStream out;
  private boolean suiteRunning = true;
  private boolean suiteSuccess = true;
  private RunMessageParser parser;

  public RunnerClient(int port, RemoteRunEventListener runnerEventListener) throws IOException {
    Socket requestSocket = getSocket(port);

    in = new ObjectInputStream(requestSocket.getInputStream());
    out = new ObjectOutputStream(requestSocket.getOutputStream());
    parser = buildParser(runnerEventListener);
  }

  private Socket getSocket(int port) throws IOException {
    Socket requestSocket = null;

    Integer attempts = 0;
    boolean keepConnecting = true;
    while (keepConnecting) {
      try {
        requestSocket = new Socket("localhost", port);
      } catch (ConnectException e) {
        if (attempts >= RECONNECT_MAX_ATTEMPTS) {
          throw e;
        }
        try {
          Thread.sleep(CONNECTION_SLEEP_MILLIS);
        } catch (InterruptedException e1) {
          return requestSocket;
        }
        attempts++;
        continue;
      }
      keepConnecting = false;
    }
    return requestSocket;
  }

  public void sendSuiteRunInfo(String runToken, String suite, String parameterizationName, Set<String> testNames,
                               Set<String> tags)
      throws IOException {
    send(new Gson().toJson(new RunMessage(RUN_SUITE, ImmutableMap.of(RUN_TOKEN_KEY, runToken,
                                                                     MUNIT_SUITE_KEY, suite,
                                                                     PARAMETERIZATION_KEY, parameterizationName,
                                                                     TEST_NAMES_KEY, collectionToString(testNames),
                                                                     TAGS_KEY, collectionToString(tags)))));
  }

  public boolean receiveAndNotify() throws IOException, ClassNotFoundException {
    do {
      String message = (String) in.readObject();
      parser.parseAndNotify(message);
    } while (suiteRunning);
    return suiteSuccess;
  }

  private void finish() {
    suiteRunning = false;
  }

  private void suiteFailed() {
    suiteSuccess = false;
  }

  private void send(String message) throws IOException {
    out.writeObject(message);
    out.flush();
  }

  private String collectionToString(Collection<String> collection) {
    return collection == null ? StringUtils.EMPTY : collection.stream().collect(Collectors.joining(SEPARATOR_TOKEN));
  }

  private RunMessageParser buildParser(RemoteRunEventListener runnerEventListener) {
    RunEventListenerContainer runnerEventListenerContainer = new RunEventListenerContainer();
    runnerEventListenerContainer.addNotificationListener((RunEventListener) runnerEventListener);
    runnerEventListenerContainer.addNotificationListener(new SuiteFinishedListener());
    runnerEventListenerContainer.addNotificationListener(new SuiteFailedListener());

    return new RunMessageParser(runnerEventListenerContainer);
  }

  private class SuiteFinishedListener implements RunEventListener {

    @Override
    public void notifySuiteUnexpectedError(String name, String stackTrace) {
      notifyUnexpectedError(stackTrace);
    }

    @Override
    public void notifySuiteEnd(String suite, String parameterization, long elapsedTime) {
      finish();
    }

    @Override
    public void notifyUnexpectedError(String stackTrace) {
      finish();
    }
  }

  private class SuiteFailedListener implements RunEventListener {

    @Override
    public void notifyTestEnd(String name, String stackTrace, TestStatus status, long elapsedTime) {
      handleStatus(status);
    }

    @Override
    public void notifyBeforeSuiteEnd(String name, String stackTrace, TestStatus status) {
      handleStatus(status);
    }

    @Override
    public void notifyAfterSuiteEnd(String name, String stackTrace, TestStatus status) {
      handleStatus(status);
    }

    @Override
    public void notifyUnexpectedError(String stackTrace) {
      handleStatus(ERROR);
    }

    @Override
    public void notifySuiteUnexpectedError(String name, String stackTrace) {
      handleStatus(ERROR);
    }

    private void handleStatus(TestStatus status) {
      if (FAILURE.equals(status) || ERROR.equals(status)) {
        suiteFailed();
      }
    }
  }

}
