/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.remote.tools.client.BAT;

import static com.google.common.net.HttpHeaders.AUTHORIZATION;
import static org.apache.commons.lang3.StringUtils.isEmpty;
import org.mule.munit.remote.tools.client.BAT.model.request.CreateExecutionRequest;
import org.mule.munit.remote.tools.client.BAT.model.request.ExecutionStatusRequest;
import org.mule.munit.remote.tools.client.BAT.model.request.TestType;
import org.mule.munit.remote.tools.client.BAT.model.response.CreateExecutionResponse;
import org.mule.munit.remote.tools.client.BAT.model.response.ExecutionResponse;
import org.mule.munit.remote.tools.client.BAT.model.response.ExecutionResultResponse;
import org.mule.munit.remote.tools.client.BAT.model.response.ExecutionStatus;
import org.mule.tools.client.authentication.AuthenticationServiceClient;
import org.mule.tools.client.authentication.model.AnypointCredential;
import org.mule.tools.client.authentication.model.ConnectedAppCredentials;
import org.mule.tools.client.authentication.model.Credentials;
import org.mule.tools.client.core.AbstractClient;

import com.google.gson.Gson;

import java.io.File;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;

import javax.ws.rs.client.Entity;
import javax.ws.rs.client.Invocation;
import javax.ws.rs.core.Response;

import org.glassfish.jersey.media.multipart.FormDataBodyPart;
import org.glassfish.jersey.media.multipart.FormDataMultiPart;
import org.glassfish.jersey.media.multipart.MultiPart;
import org.glassfish.jersey.media.multipart.file.FileDataBodyPart;

/**
 * Client for BAT Executions services
 *
 * @author Mulesoft Inc.
 * @since 2.3.0
 */
public abstract class BATClientBase extends AbstractClient {

  public static final String EXECUTION_BASE_URL = "/apitesting/xapi/v1/organizations/%s";

  public static final String EXECUTION_RESULT_BASE_URL = "/apitesting/worker/api/v1/organizations/%s/executions";

  public static final String RESULTS_PATH = "%s/results";
  public static final String EXECUTION_RESULT_PATH = "/executions/%s/result";
  public static final String EXECUTIONS_PATH = "/executions";
  public static final String EXECUTION_ID_PATH = "/executions/%s";
  public static final String BEARER_TOKEN = "Bearer";

  protected String bearerToken;
  private String organizationId;
  private String baseExecutionUrl;
  private String baseResultURL;

  private final String baseUri;

  public BATClientBase(String baseUri) {
    this.baseUri = baseUri;
  }

  @Override
  protected void init() {}

  /***
   * Creates new Execution
   * @param testType type of test for the Execution
   * @return the Execution info
   */
  public CreateExecutionResponse createExecution(TestType testType) {
    setUp();

    CreateExecutionRequest request = new CreateExecutionRequest();
    request.setTestType(testType.name());

    Response response = post(baseExecutionUrl, EXECUTIONS_PATH, new Gson().toJson(request));

    checkResponseStatus(response);

    return response.readEntity(CreateExecutionResponse.class);
  }

  /***
   * Retrieves the Execution info
   * @param executionId Id of the Execution
   * @return the Execution info
   */
  public ExecutionResponse getExecution(String executionId) {
    setUp();

    Response response = get(
                            baseExecutionUrl,
                            String.format(EXECUTION_ID_PATH, executionId));

    checkResponseStatus(response);

    return new Gson().fromJson(response.readEntity(String.class), ExecutionResponse.class);
  }

  /***
   * Retrieves the test result from an Execution
   * @param executionId Id of the Execution
   * @return the test result saved for the Execution
   */
  public ExecutionResultResponse getExecutionResult(String executionId) {
    setUp();

    Response response = get(
                            baseExecutionUrl,
                            String.format(EXECUTION_RESULT_PATH, executionId));

    checkResponseStatus(response);

    return new Gson().fromJson(response.readEntity(String.class), ExecutionResultResponse.class);
  }

  /**
   * Save the test result to an Execution
   *
   * @param executionId Id of the Execution
   * @param file        that contains the result to be uploaded
   * @throws IOException if reading file action fails
   */
  public void saveExecutionResult(String executionId, File file) throws IOException {
    setUp();

    String fileContent = new String(Files.readAllBytes(file.toPath()), StandardCharsets.UTF_8);
    FileDataBodyPart filePart = new FileDataBodyPart("result", file);
    FormDataBodyPart appInfoJsonPart = new FormDataBodyPart("appInfoJson", fileContent);
    MultiPart multipart = (new FormDataMultiPart()).bodyPart(filePart).bodyPart(appInfoJsonPart);
    Entity<MultiPart> entity = Entity.entity(multipart, multipart.getMediaType());
    Response response = this.put(baseResultURL, String.format(RESULTS_PATH, executionId), entity);

    checkResponseStatus(response);
  }

  /**
   * Change the status of an Execution
   *
   * @param executionId Id of the Execution
   * @param status      the new status for the Execution
   * @return Execution object with new status
   */
  public ExecutionResponse setExecutionStatus(String executionId, ExecutionStatus status) {
    setUp();

    ExecutionStatusRequest request = new ExecutionStatusRequest();
    request.setStatus(status.name());
    Response response = this.put(baseResultURL,
                                 String.format("%s", executionId), request);

    checkResponseStatus(response);

    return new Gson().fromJson(response.readEntity(String.class), ExecutionResponse.class);
  }

  @Override
  protected void configureRequest(Invocation.Builder builder) {
    super.configureRequest(builder);
    builder.header(AUTHORIZATION, String.format("%s %s", BEARER_TOKEN, this.bearerToken));
  }

  private void setUp() {
    if (!isEmpty(organizationId)) {
      return;
    }

    this.bearerToken = getBearerToken();
    this.organizationId = getOrganizationId();
    this.baseExecutionUrl = String.format(baseUri.concat(EXECUTION_BASE_URL), this.organizationId);
    this.baseResultURL = String.format(baseUri.concat(EXECUTION_RESULT_BASE_URL), this.organizationId);
  }

  protected abstract String getBearerToken();

  protected abstract String getOrganizationId();

  public abstract void renewToken();

  public String getBaseUri() {
    return baseUri;
  }
}
