/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.remote.container.cloudhub;

import static com.google.common.base.Preconditions.checkArgument;
import static org.mule.runtime.core.api.config.MuleDeploymentProperties.MULE_LAZY_CONNECTIONS_DEPLOYMENT_PROPERTY;
import static org.mule.runtime.core.api.config.MuleDeploymentProperties.MULE_LAZY_INIT_DEPLOYMENT_PROPERTY;
import org.mule.munit.common.exception.MunitError;
import org.mule.munit.common.protocol.listeners.RemoteRunEventListener;
import org.mule.munit.common.protocol.listeners.RunEventListener;
import org.mule.munit.common.protocol.message.RunMessageParser;
import org.mule.munit.remote.api.configuration.CloudHubContainerConfiguration;
import org.mule.munit.remote.api.configuration.RunConfiguration;
import org.mule.munit.remote.container.Container;
import org.mule.munit.remote.container.SuiteRunDispatcher;
import org.mule.munit.remote.container.cloudhub.logging.CloudHubLoggingTask;
import org.mule.munit.remote.exception.DeploymentException;
import org.mule.munit.remote.logging.MunitDeployerLog;
import org.mule.munit.remote.tools.client.BAT.BATClientBase;
import org.mule.munit.remote.tools.client.BAT.model.response.ExecutionResponse;
import org.mule.munit.remote.tools.client.BAT.model.response.ExecutionResultResponse;
import org.mule.munit.remote.tools.client.BAT.model.response.ExecutionStatus;
import org.mule.tools.client.OperationRetrier;
import org.mule.tools.client.cloudhub.CloudHubClient;
import org.mule.tools.client.core.exception.ClientException;
import org.mule.tools.deployment.DefaultDeployer;
import org.mule.tools.deployment.Deployer;
import org.mule.tools.model.anypoint.AnypointDeployment;
import org.mule.tools.model.anypoint.CloudHubDeployment;
import org.mule.tools.utils.DeployerLog;

import com.google.common.io.ByteStreams;
import com.google.gson.Gson;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.nio.file.Path;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.TimeoutException;
import java.util.jar.JarEntry;
import java.util.jar.JarOutputStream;

import javax.ws.rs.core.Response;

import org.apache.logging.log4j.util.Strings;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Dispatches a Suite Run to CloudHub with a given {@link RunConfiguration}
 *
 * @author Mulesoft Inc.
 * @since 2.2.0
 */
public class CloudHubRunDispatcher implements SuiteRunDispatcher {

  private transient Logger logger = LoggerFactory.getLogger(this.getClass());

  public static final long UNDEPLOYMENT_RETRIER_SLEEP_TIME = 2000;
  public static final int UNDEPLOYMENT_RETRIER_ATTEMPS = (int) ((60 * 2 * 1000) / UNDEPLOYMENT_RETRIER_SLEEP_TIME);

  public static final int DEFAULT_TIMEOUT = (60 * 30 * 1000);
  public static final long RESULT_RETRIER_SLEEP_TIME = 10 * 1000;
  public static final int RESULT_RETRIER_ATTEMPS = (int) (DEFAULT_TIMEOUT / UNDEPLOYMENT_RETRIER_SLEEP_TIME);

  public static final String CLOUDHUB_DASHBOARD_URL_FORMAT = "cloudhub/#/console/applications/cloudhub/%s/logging";

  protected Deployer deployer;
  protected RunConfiguration runConfiguration;
  protected CloudHubDeployment deployment;
  protected DeployerLog deployerLog;
  private final CloudHubContainerConfiguration containerConfiguration;
  private final CloudHubClient cloudHubClient;
  private final BATClientBase batClient;

  public CloudHubRunDispatcher(RunConfiguration runConfiguration, CloudHubClient cloudHubClient, BATClientBase batClient) {
    checkArgument(runConfiguration != null, "runConfiguration must not be null");
    checkArgument(runConfiguration.getContainerConfiguration() instanceof CloudHubContainerConfiguration,
                  "containerConfiguration must be instance of CloudHubContainerConfiguration");
    checkArgument(cloudHubClient != null, "cloudHubClient must not be null");
    checkArgument(batClient != null, "batClient must not be null");

    this.runConfiguration = runConfiguration;
    this.cloudHubClient = cloudHubClient;
    this.containerConfiguration = (CloudHubContainerConfiguration) runConfiguration.getContainerConfiguration();

    checkArgument(containerConfiguration.getDeploymentConfiguration().getDeployment() instanceof CloudHubDeployment,
                  "deployment must be instance of CloudHubDeployment");

    CloudHubContainerConfiguration containerConfiguration =
        (CloudHubContainerConfiguration) runConfiguration.getContainerConfiguration();
    this.deployment = (CloudHubDeployment) containerConfiguration.getDeploymentConfiguration().getDeployment();
    this.deployerLog = new MunitDeployerLog(containerConfiguration.isDebugEnabled());
    this.batClient = batClient;
  }

  @Override
  public void runSuites(RemoteRunEventListener listener) throws DeploymentException {
    deployApplication();

    try {
      String urlDashboard = deployment.getUri() + String.format(CLOUDHUB_DASHBOARD_URL_FORMAT, deployment.getApplicationName());
      logger.info("Running MUnit on CloudHub. Track the progress on: " + urlDashboard);
      OperationRetrier retrier = new OperationRetrier();
      retrier.setAttempts(RESULT_RETRIER_ATTEMPS);
      retrier.setSleepTime(RESULT_RETRIER_SLEEP_TIME);
      retrier.retry(() -> {
        try {
          String executionId = containerConfiguration.getExecutionId();

          ExecutionResponse response = this.batClient.getExecution(executionId);

          if (response.getStatus() == ExecutionStatus.ENDED) {

            ExecutionResultResponse result = this.batClient.getExecutionResult(executionId);

            RunMessageParser parser = new RunMessageParser((RunEventListener) listener);

            Gson gson = new Gson();
            result.getData().forEach(runMessage -> parser.parseAndNotify(gson.toJson(runMessage)));
            return false;
          } else if (response.getStatus() == ExecutionStatus.ERROR) {
            throw new MunitError("An error occurred running MUnit on CloudHub.");
          }
          return true;
        } catch (ClientException httpException) {
          if (httpException.getStatusCode() == Response.Status.UNAUTHORIZED.getStatusCode()) {
            batClient.renewToken();
            return true;
          } else {
            throw new MunitError("An error occurred trying to retrieve the Execution.");
          }
        }
      });
    } catch (InterruptedException | TimeoutException e) {
      throw new MunitError("Tests timeout after " + DEFAULT_TIMEOUT + " milliseconds");
    } finally {
      this.cloudHubClient.renewToken();
      CloudHubLoggingTask loggingTask = getCloudhubLogger();
      loggingTask.log();
      undeployApplication();
    }
  }

  protected void deployApplication() throws DeploymentException {
    logger.info("CloudHub deploying application");

    try {
      Path munitWorkingPath = new File(runConfiguration.getContainerConfiguration().getMunitWorkingDirectoryPath()).toPath();
      String jarFilename = runConfiguration.getProjectName() + ".jar";

      replaceLog4j();

      generateJar(munitWorkingPath.resolve(runConfiguration.getProjectName()).toFile(),
                  munitWorkingPath.resolve(jarFilename).toFile());

      deployment
          .setArtifact(new File(runConfiguration.getContainerConfiguration().getMunitWorkingDirectoryPath()).toPath()
              .resolve(jarFilename).toFile());

      this.deployer = getDeployer();
      setProperties(deployment);
      deployment.setSkipDeploymentVerification(true);
      deployer.deploy();
    } catch (Exception e) {
      throw new DeploymentException("An error occurred while deploying application: " + e.getMessage(), e);
    }
  }

  protected void undeployApplication() {
    logger.info("CloudHub undeploying application");

    try {
      if (deployer != null) {
        deployer.undeploy();
        waitUndeploymentFinish();
      }
    } catch (org.mule.tools.client.core.exception.DeploymentException | ClientException e) {
      logger.error("An error occurred while undeploying application: " + e.getMessage());
    }
  }

  protected void generateJar(File srcFolder, File destJarFile) throws IOException {
    try (FileOutputStream fileWriter = new FileOutputStream(destJarFile);
        JarOutputStream jar = new JarOutputStream(fileWriter)) {
      for (File file : srcFolder.listFiles()) {
        addFileToJar(Strings.EMPTY, file, jar);
      }
    }
  }

  private void addFileToJar(String prefix, File srcFile, JarOutputStream jar) throws IOException {
    if (srcFile.isDirectory()) {
      for (File file : srcFile.listFiles()) {
        String newPrefix = prefix + srcFile.getName() + "/";
        addFileToJar(newPrefix, file, jar);
      }
    } else {
      try (FileInputStream in = new FileInputStream(srcFile)) {
        jar.putNextEntry(new JarEntry(prefix + srcFile.getName()));
        ByteStreams.copy(in, jar);
      }
    }
  }

  protected void replaceLog4j() {
    Path munitWorkingPath = new File(runConfiguration.getContainerConfiguration().getMunitWorkingDirectoryPath()).toPath();
    Path appWorkingPath = munitWorkingPath.resolve(runConfiguration.getProjectName());

    appWorkingPath.resolve("log4j2.xml").toFile().delete();
    appWorkingPath.resolve("log4j2-test.xml").toFile().renameTo(appWorkingPath.resolve("log4j2.xml").toFile());
  }

  private void setProperties(CloudHubDeployment deployment) {
    Map<String, String> properties = new HashMap<>();

    properties.put(MULE_LAZY_CONNECTIONS_DEPLOYMENT_PROPERTY, Container.enableLazyConnections().toString());
    properties.put(MULE_LAZY_INIT_DEPLOYMENT_PROPERTY, Container.enableLazyInitialization().toString());

    if (containerConfiguration.getSystemPropertyVariables() != null
        && !containerConfiguration.getSystemPropertyVariables().isEmpty()) {
      properties.putAll(containerConfiguration.getSystemPropertyVariables());
    }

    if (containerConfiguration.getEnvironmentVariables() != null && !containerConfiguration.getEnvironmentVariables().isEmpty()) {
      properties.putAll(containerConfiguration.getEnvironmentVariables());
    }

    AnypointDeployment anypointDeployment = deployment;

    if (anypointDeployment.getProperties() != null) {
      properties.putAll(anypointDeployment.getProperties());
    }

    anypointDeployment.setProperties(properties);
  }

  private void waitUndeploymentFinish() {
    OperationRetrier retrier = new OperationRetrier();
    retrier.setAttempts(UNDEPLOYMENT_RETRIER_ATTEMPS);
    retrier.setSleepTime(UNDEPLOYMENT_RETRIER_SLEEP_TIME);

    try {
      retrier.retry(() -> cloudHubClient.isDomainAvailable(deployment.getApplicationName()));
    } catch (InterruptedException | TimeoutException e) {
      logger.debug("waitUndeploymentFinish timeout.");
    }
  }

  protected Deployer getDeployer() throws org.mule.tools.client.core.exception.DeploymentException {
    return new DefaultDeployer(deployment, deployerLog);
  }

  private CloudHubLoggingTask getCloudhubLogger() {
    return new CloudHubLoggingTask(deployerLog, this.cloudHubClient,
                                   cloudHubClient.getApplications(deployment.getApplicationName()));
  }
}
