/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.remote.container;

import static org.mule.maven.client.api.model.MavenConfiguration.newMavenConfigurationBuilder;
import static org.mule.maven.client.api.model.RemoteRepository.newRemoteRepositoryBuilder;
import static org.mule.munit.remote.FolderNames.CONTAINER;
import static org.mule.munit.remote.container.BundleDescriptorFactory.buildServerPluginDescriptors;
import static org.mule.munit.remote.logging.Log4jConfigurator.getDefaultLog4jConfiguration;
import org.mule.maven.client.api.model.BundleDescriptor;
import org.mule.maven.client.api.model.MavenConfiguration;
import org.mule.munit.remote.api.configuration.EmbeddedContainerConfiguration;
import org.mule.munit.remote.api.configuration.Repository;
import org.mule.munit.remote.api.configuration.RunConfiguration;
import org.mule.munit.remote.api.configuration.ServerPluginConfiguration;
import org.mule.runtime.module.embedded.api.Product;

import java.io.File;
import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URI;
import java.net.URL;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.List;

import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * It knows how to build a embedded container, used to start Mule programmatically
 *
 * @author Mulesoft Inc.
 * @since 2.0.0
 */
public class EmbeddedContainerFactory {

  protected transient Logger logger = LoggerFactory.getLogger(this.getClass());

  protected static final String MAVEN_CENTRAL_ID = "maven-central";
  protected static final String MULESOFT_PUBLIC_ID = "mulesoft-public";

  protected static final String MAVEN_CENTRAL_URL = "https://repo.maven.apache.org/maven2/";
  protected static final String MULESOFT_PUBLIC_URL = "https://repository.mulesoft.org/nexus/content/repositories/public";

  public Container createContainer(RunConfiguration runConfiguration) throws ContainerFactoryException {
    EmbeddedContainerConfiguration configuration = (EmbeddedContainerConfiguration) runConfiguration.getContainerConfiguration();
    Path munitWorkingDirectoryPath = Paths.get(configuration.getMunitWorkingDirectoryPath());
    File containerBaseFolder = munitWorkingDirectoryPath.resolve(CONTAINER.value()).toFile();

    String runtimeVersion = configuration.getRuntimeId();
    String product = configuration.getProduct();
    URI log4JConfiguration = getLog4JConfiguration(configuration);
    BundleDescriptor[] serverPlugins = buildServerPluginDescriptors(configuration.getServerPluginConfigurations());
    loadServerPluginProperties(configuration.getServerPluginConfigurations());

    org.mule.runtime.module.embedded.api.ContainerConfiguration containerConfiguration =
        org.mule.runtime.module.embedded.api.ContainerConfiguration.builder()
            .containerFolder(containerBaseFolder)
            .serverPlugins(serverPlugins)
            .build();

    return new EmbeddedContainer(org.mule.runtime.module.embedded.api.EmbeddedContainer.builder()
        .muleVersion(runtimeVersion)
        .product(Product.valueOf(product))
        .log4jConfigurationFile(log4JConfiguration)
        .containerConfiguration(containerConfiguration)
        .mavenConfiguration(getMavenConfiguration(configuration.getMavenConfiguration()))
        .build(), runConfiguration);

  }

  protected URI getLog4JConfiguration(EmbeddedContainerConfiguration runConfig) throws ContainerFactoryException {
    try {
      String log4JConfigurationFilePath = runConfig.getLog4JConfigurationFilePath();
      if (StringUtils.isNotBlank(log4JConfigurationFilePath)) {
        File log4JConfigurationFile = Paths.get(log4JConfigurationFilePath).toFile();
        if (log4JConfigurationFile != null && log4JConfigurationFile.exists()) {
          return log4JConfigurationFile.toURI();
        }
      }
      return getDefaultLog4jConfiguration();
    } catch (IOException e) {
      throw new ContainerFactoryException("Unable to obtainer log4j configuration file to create container", e);
    }
  }

  protected void loadServerPluginProperties(List<ServerPluginConfiguration> pluginConfig) {
    if (pluginConfig != null) {
      pluginConfig.stream().filter(config -> config.getProperties() != null)
          .forEach(config -> System.getProperties().putAll(config.getProperties()));
    }
  }

  protected MavenConfiguration getMavenConfiguration(org.mule.munit.remote.api.configuration.MavenConfiguration configuration)
      throws ContainerFactoryException {
    MavenConfiguration.MavenConfigurationBuilder builder = newMavenConfigurationBuilder();

    if (configuration.getMavenRepositoryDirectoryPath() != null) {
      File mavenRepository = Paths.get(configuration.getMavenRepositoryDirectoryPath()).toFile();
      if (mavenRepository.exists()) {
        builder.localMavenRepositoryLocation(mavenRepository);
      }
    }

    if (configuration.getSettingsXmlFilePath() != null) {
      File settingsXml = Paths.get(configuration.getSettingsXmlFilePath()).toFile();
      if (settingsXml.exists()) {
        builder.userSettingsLocation(settingsXml);
      }
    }

    if (configuration.getGlobalSettingsXmlFilePath() != null) {
      File globalSettingsXml = Paths.get(configuration.getGlobalSettingsXmlFilePath()).toFile();
      if (globalSettingsXml.exists()) {
        builder.globalSettingsLocation(globalSettingsXml);
      }
    }

    if (configuration.getSecuritySettingsFilePath() != null) {
      File securitySettingsFile =
          Paths.get(configuration.getSecuritySettingsFilePath()).toFile();
      if (securitySettingsFile.exists()) {
        builder.settingsSecurityLocation(securitySettingsFile);
      }
    }

    builder.activeProfiles(configuration.getActiveProfiles());
    builder.offlineMode(configuration.getOfflineMode());
    builder.ignoreArtifactDescriptorRepositories(configuration.getIgnoreArtifactDescriptorRepositories());

    if (configuration.getForcePolicyUpdate()) {
      builder.forcePolicyUpdateNever(false);
    } else {
      builder.forcePolicyUpdateNever(true);
    }

    addDefaultRepositories(builder, configuration);
    return builder.build();
  }

  protected void addDefaultRepositories(MavenConfiguration.MavenConfigurationBuilder builder,
                                        org.mule.munit.remote.api.configuration.MavenConfiguration configuration)
      throws ContainerFactoryException {

    try {
      builder.remoteRepository(newRemoteRepositoryBuilder().id(MAVEN_CENTRAL_ID).url(new URL(MAVEN_CENTRAL_URL))
          .build());
      builder.remoteRepository(newRemoteRepositoryBuilder().id(MULESOFT_PUBLIC_ID).url(new URL(MULESOFT_PUBLIC_URL))
          .build());
    } catch (MalformedURLException e) {
      throw new ContainerFactoryException("Malformed URLs when adding default repositories", e);
    }

    for (Repository repository : configuration.getRemoteRepositories()) {
      if (!repository.getId().equals(MAVEN_CENTRAL_ID) &&
          !repository.getId().equals(MULESOFT_PUBLIC_ID)) {
        try {
          builder.remoteRepository(newRemoteRepositoryBuilder().id(repository.getId()).url(new URL(repository.getUrl()))
              .build());
        } catch (MalformedURLException e) {
          logger.warn("Repository " + repository.getUrl()
              + " could not be added as a repository to download dependencies for the container.");
        }
      }
    }
  }
}
