/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.remote.container;

import static org.apache.commons.lang3.StringUtils.isNotBlank;

import org.mule.munit.common.properties.MUnitUserPropertiesManager;
import org.mule.munit.remote.api.configuration.RunConfiguration;
import org.mule.munit.remote.exception.DeploymentException;
import org.mule.munit.remote.exception.DeploymentExceptionThrower;
import org.mule.runtime.module.embedded.api.ArtifactConfiguration;
import org.mule.runtime.module.embedded.api.DeploymentConfiguration;
import org.mule.runtime.module.embedded.api.DeploymentService;

import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.Map;


/**
 * Wrapper for the {@link org.mule.runtime.module.embedded.api.EmbeddedContainer} that provides a friendlier API
 *
 * @author Mulesoft Inc.
 * @since 2.2.0
 */
public class EmbeddedContainer implements Container {

  private final org.mule.runtime.module.embedded.api.EmbeddedContainer container;
  private final RunConfiguration runConfiguration;

  public EmbeddedContainer(org.mule.runtime.module.embedded.api.EmbeddedContainer container, RunConfiguration runConfiguration) {
    this.container = container;
    this.runConfiguration = runConfiguration;
  }

  @Override
  public void deployDomain() throws DeploymentException {
    String domainLocation = runConfiguration.getDomainLocation();
    if (isNotBlank(domainLocation)) {
      Path artifactLocation = Paths.get(domainLocation);
      ArtifactConfiguration artifactConfiguration =
          getArtifactConfiguration(artifactLocation, buildDeploymentConfiguration(true));
      DeploymentService deploymentService = container.getDeploymentService();
      try {
        deploymentService.deployDomain(artifactConfiguration);
      } catch (RuntimeException e) {
        DeploymentExceptionThrower.throwIfMatches(e);
      }
    }
  }

  @Override
  public void deployApplication(boolean enableXmlValidations, Map<String, Object> properties)
      throws DeploymentException {
    String applicationName = runConfiguration.getProjectName();
    String munitWorkingDirectoryPath = runConfiguration.getContainerConfiguration().getMunitWorkingDirectoryPath();
    Path artifactLocation = Paths.get(munitWorkingDirectoryPath, applicationName);
    ArtifactConfiguration artifactConfiguration =
        getArtifactConfiguration(artifactLocation, buildDeploymentConfiguration(enableXmlValidations));
    DeploymentService deploymentService = container.getDeploymentService();
    MUnitUserPropertiesManager.addUserPropertiesToSystem(properties);
    try {
      deploymentService.deployApplication(artifactConfiguration);
    } catch (RuntimeException e) {
      DeploymentExceptionThrower.throwIfMatches(e);
    }
  }

  @Override
  public void undeployApplication(Map<String, Object> systemProperties) {
    if (runConfiguration.getClearParameters()) {
      MUnitUserPropertiesManager.removeProperties(systemProperties);
    }
    container.getDeploymentService().undeployApplication(runConfiguration.getProjectName());
  }

  private ArtifactConfiguration getArtifactConfiguration(Path artifactLocation, DeploymentConfiguration deploymentConfiguration) {
    return ArtifactConfiguration.builder()
        .deploymentConfiguration(deploymentConfiguration)
        .artifactLocation(artifactLocation.toFile())
        .build();
  }

  protected DeploymentConfiguration buildDeploymentConfiguration(boolean enableXmlValidations) {
    return DeploymentConfiguration.builder()
        .lazyConnectionsEnabled(Container.enableLazyConnections())
        .lazyInitialization(Container.enableLazyInitialization())
        .xmlValidations(enableXmlValidations)
        .addArtifactAstToRegistry(true)
        .build();
  }

  @Override
  public void start() {
    container.start();
  }

  @Override
  public void stop() {
    container.stop();
  }

  /**
   * @return the version if the Mule runtime this embedded container is running.
   */
  @Override
  public String getMuleContainerVersion() {
    return container.getMuleContainerVersion();
  }

}
