/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
/*
 * Copyright (c) 2015 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.remote.api.configuration;

import java.io.IOException;
import java.util.Map;

import com.google.gson.Gson;
import com.google.gson.TypeAdapter;
import com.google.gson.annotations.JsonAdapter;
import com.google.gson.stream.JsonReader;
import com.google.gson.stream.JsonWriter;

/**
 * Defines the common interface for base Container Configuration
 *
 * @author Mulesoft Inc.
 * @since 2.3.0
 */
@JsonAdapter(ContainerConfiguration.ContainerConfigurationAdapter.class)
public abstract class ContainerConfiguration {

  private String runtimeId;
  private String product;
  private String munitWorkingDirectoryPath;
  private String log4JConfigurationFilePath;
  protected String type;
  private Map<String, String> systemPropertyVariables;
  private Map<String, String> environmentVariables;

  public ContainerConfiguration(String type, String runtimeId, String product, String munitWorkingDirectoryPath,
                                String log4JConfigurationFilePath, Map<String, String> systemPropertyVariables,
                                Map<String, String> environmentVariables) {
    this.munitWorkingDirectoryPath = munitWorkingDirectoryPath;
    this.log4JConfigurationFilePath = log4JConfigurationFilePath;
    this.runtimeId = runtimeId;
    this.product = product;
    this.type = type;
    this.systemPropertyVariables = systemPropertyVariables;
    this.environmentVariables = environmentVariables;
  }

  public ContainerConfiguration() {};

  public abstract DeploymentType getType();

  public String getMunitWorkingDirectoryPath() {
    return munitWorkingDirectoryPath;
  }

  public String getLog4JConfigurationFilePath() {
    return log4JConfigurationFilePath;
  }

  public String getRuntimeId() {
    return runtimeId;
  }

  public String getProduct() {
    return product;
  }

  public Map<String, String> getSystemPropertyVariables() {
    return systemPropertyVariables;
  }

  public Map<String, String> getEnvironmentVariables() {
    return environmentVariables;
  }

  /**
   * Container Configuration Builder
   *
   * @author Mulesoft Inc.
   * @since 2.3.0
   */
  public interface ContainerConfigurationBuilder {

    ContainerConfiguration build();
  }

  static class ContainerConfigurationAdapter extends TypeAdapter<ContainerConfiguration> {

    private static final String TYPE = "type";
    private static final String VALUE = "value";
    private static final Gson gson = new Gson();

    @Override
    public void write(JsonWriter out, ContainerConfiguration containerConfiguration) throws IOException {
      out.beginObject();
      out.name(TYPE);
      out.value(containerConfiguration.getType().name());
      out.name(VALUE);

      switch (containerConfiguration.getType()) {
        case CLOUDHUB:
          gson.toJson(containerConfiguration, CloudHubContainerConfiguration.class, out);
          break;
        case EMBEDDED:
        default:
          gson.toJson(containerConfiguration, EmbeddedContainerConfiguration.class, out);
          break;
      }
      out.endObject();
    }

    @Override
    public ContainerConfiguration read(JsonReader in) throws IOException {
      in.beginObject();
      if (!TYPE.equals(in.nextName())) {
        throw new IOException("Container configuration serialized does not have the type field");
      }
      String containerTypeValue = in.nextString();
      DeploymentType containerType = DeploymentType.fromValue(containerTypeValue);
      if (containerType == null) {
        throw new IOException("Container configuration type not found, value was " + containerTypeValue);
      }

      if (!VALUE.equals(in.nextName())) {
        throw new IOException("Container configuration serialized does not have the value field");
      }

      ContainerConfiguration containerConfiguration;

      switch (containerType) {
        case CLOUDHUB:
          containerConfiguration = gson.fromJson(in, CloudHubContainerConfiguration.class);
          break;
        case EMBEDDED:
        default:
          containerConfiguration = gson.fromJson(in, EmbeddedContainerConfiguration.class);
          break;
      }
      in.endObject();
      return containerConfiguration;
    }
  }
}
