/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.remote.properties;

import static java.util.Collections.emptySet;

import org.mule.munit.common.exception.MunitError;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

import org.jdom2.Attribute;
import org.jdom2.Document;
import org.jdom2.Element;
import org.jdom2.JDOMException;
import org.jdom2.Namespace;
import org.jdom2.input.SAXBuilder;
import org.mule.munit.remote.runtime.utils.Product;
import org.yaml.snakeyaml.Yaml;

/**
 * Parses the parameterization configuration and returns the parameterized suites, if present
 *
 * @author Mulesoft Inc.
 * @since 2.2.0
 */
public class SuiteConfigParser {

  // Schema names and namespaces
  private static final Namespace MUNIT_NAMESPACE = Namespace.getNamespace("munit", "http://www.mulesoft.org/schema/mule/munit");
  private static final String CONFIG_NAME = "config";
  private static final String PARAMETERIZATIONS_NAME = "parameterizations";
  private static final String PARAMETERIZATION_NAME = "parameterization";
  private static final String PARAMETERS_NAME = "parameters";
  private static final String PARAMETER_NAME = "parameter";
  private static final String PROPERTY_NAME_NAME = "propertyName";
  private static final String VALUE_NAME = "value";
  private static final String NAME_NAME = "name";
  private static final String IGNORE_FIELD = "ignore";
  private static final String MIN_MULE_VERSION_FIELD = "minMuleVersion";
  private static final String REQUIRED_PRODUCT_FIELD = "requiredProduct";

  private Element config;
  private File suiteFile;
  private File resourcesFolder;

  public SuiteConfigParser(File suiteFile, File resourcesFolder) {
    this.suiteFile = suiteFile;
    this.resourcesFolder = resourcesFolder;

    try {
      SAXBuilder saxBuilder = XmlUtils.createSecureSAXBuilder();
      Document document = saxBuilder.build(suiteFile);
      Element rootElement = document.getRootElement();
      config = rootElement.getChild(CONFIG_NAME, MUNIT_NAMESPACE);
    } catch (JDOMException | IOException e) {
      config = null;
    }
  }

  public Optional<String> getMinMuleVersion() {
    if (config == null) {
      return Optional.empty();
    }
    return Optional.ofNullable(config.getAttributeValue(MIN_MULE_VERSION_FIELD));
  }

  public Optional<Product> getRequiredProduct() {
    if (config == null || config.getAttributeValue(REQUIRED_PRODUCT_FIELD) == null) {
      return Optional.empty();
    }
    return Optional.of(Product.valueOf(config.getAttributeValue(REQUIRED_PRODUCT_FIELD)));
  }

  public boolean isIgnored() {
    if (config == null) {
      return false;
    }
    return Boolean.valueOf(config.getAttributeValue(IGNORE_FIELD));
  }

  public Set<Parameterization> parseParameterizations() {
    if (config == null) {
      return emptySet();
    }
    Element parameterizations = config.getChild(PARAMETERIZATIONS_NAME, MUNIT_NAMESPACE);
    if (parameterizations == null) {
      return emptySet();
    }

    Set<Parameterization> parameterizedSuites = new HashSet<>();

    Attribute fileAttribute = parameterizations.getAttribute("file");
    if (fileAttribute != null) {
      parameterizedSuites.addAll(collectParameterizedSuitesFromFile(fileAttribute.getValue()));
    }

    parameterizedSuites.addAll(collectParameterizedSuites(parameterizations.getChildren(PARAMETERIZATION_NAME, MUNIT_NAMESPACE)));

    Map<String, Parameterization> mapParameterizedSuites = new HashMap<>();

    for (Parameterization param : parameterizedSuites) {
      mapParameterizedSuites.put(param.getParameterizationName(), param);
    }

    return new HashSet<>(mapParameterizedSuites.values());
  }

  private Set<Parameterization> collectParameterizedSuites(List<Element> parameterizations) {
    Set<Parameterization> parameterizedSuites = new HashSet<>();
    for (Element parameterization : parameterizations) {
      Map<String, String> parametersMap = new HashMap<>();
      Element parameters = parameterization.getChild(PARAMETERS_NAME, MUNIT_NAMESPACE);
      if (parameters != null) {
        List<Element> parameterChildren = parameters.getChildren(PARAMETER_NAME, MUNIT_NAMESPACE);
        for (Element parameter : parameterChildren) {
          parametersMap.put(parameter.getAttributeValue(PROPERTY_NAME_NAME), parameter.getAttributeValue(VALUE_NAME));
        }
      }
      String name = parameterization.getAttributeValue(NAME_NAME);
      parameterizedSuites.add(new Parameterization(name, parametersMap));
    }
    return parameterizedSuites;
  }

  @SuppressWarnings("unchecked")
  private Set<Parameterization> collectParameterizedSuitesFromFile(String fileName) {
    Set<Parameterization> parameterizedSuites = new HashSet<>();

    try {
      Yaml yml = new Yaml();
      File file = new File(resourcesFolder, fileName);

      FileInputStream fileStream = new FileInputStream(file);
      Map<String, Map<String, String>> props = ((Map<String, Map<String, Object>>) yml.load(fileStream))
          .entrySet().stream()
          .collect(Collectors.toMap(
                                    Map.Entry::getKey,
                                    entry -> entry.getValue().entrySet().stream().collect(Collectors
                                        .toMap(Map.Entry::getKey, entryValue -> String.valueOf(entryValue.getValue())))));

      for (String key : props.keySet()) {
        Map<String, String> parametersMap = new HashMap<>();
        parametersMap.putAll(props.get(key));
        parameterizedSuites.add(new Parameterization(key, parametersMap));
      }

    } catch (FileNotFoundException e) {
      throw new MunitError(String.format("Parameterization file %s not found for suite %s.", fileName, this.suiteFile.getName()));
    } catch (Exception e) {
      throw new MunitError(String.format("Unable to parse parameterization file %s for suite %s.", fileName,
                                         this.suiteFile.getName()));
    }

    return parameterizedSuites;
  }

}
