/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.remote;

import com.evanlennick.retry4j.CallExecutorBuilder;
import com.evanlennick.retry4j.config.RetryConfig;
import com.evanlennick.retry4j.config.RetryConfigBuilder;
import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.LogManager;

import java.io.File;
import java.io.RandomAccessFile;
import java.nio.channels.FileChannel;
import java.nio.channels.FileLock;

import static java.time.temporal.ChronoUnit.SECONDS;

/**+
 * @author Mulesoft Inc.
 */
public class MuleDxMunitLock {

  private static final Logger LOGGER = LogManager.getLogger(MuleDxMunitLock.class);

  private final FileChannel channel;
  final File file;
  private FileLock lock;

  public MuleDxMunitLock(String lockId) {
    file = new File(System.getProperty("java.io.tmpdir"), lockId + ".tmp");
    try {
      this.channel = new RandomAccessFile(file, "rw").getChannel();
    } catch (Exception exception) {
      LOGGER.error(exception.getMessage(), exception);
      throw new RuntimeException(exception);
    }
  }

  public void lock() {
    lock(25);
  }

  public void lock(int tries) {
    RetryConfig config = new RetryConfigBuilder()
        .retryOnAnyException()
        .withMaxNumberOfTries(tries)
        .withDelayBetweenTries(2, SECONDS)
        .withExponentialBackoff()
        .build();
    new CallExecutorBuilder<Void>()
        .config(config)
        .build()
        .execute(() -> {
          try {
            LOGGER.debug("Attempting to connect lock file." + file.getAbsolutePath());
            lock = channel.tryLock();
            if (lock == null) {
              throw new IllegalStateException();
            }
          } catch (IllegalStateException exception) {
            // This exception is throws is the file is already locked
            throw exception;
          } catch (Exception exception) {
            LOGGER.error("Unexpected exception. Lock not be possible", exception);
          }
          return null;
        });
  }

  public void release() {
    try {
      if (lock != null) {
        lock.release();
        channel.close();
      }
    } catch (Exception exception) {
      LOGGER.error(exception.getMessage(), exception);
      throw new RuntimeException(exception);
    }
  }
}
