/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.remote.container.model;

import static org.mule.munit.common.util.Preconditions.checkArgument;

import java.util.HashMap;
import java.util.Map;
import java.util.Set;

/**
 * Represents a suite to be run
 *
 * @author Mulesoft Inc.
 * @since 2.2.0
 */
public class SuiteDeployment {

  private Set<SuiteRun> suiteRuns;
  private Map<String, Object> systemProperties;
  private boolean enableXmlValidations;

  private SuiteDeployment(Set<SuiteRun> suiteRuns, Map<String, Object> systemProperties, boolean enableXmlValidations) {
    this.suiteRuns = suiteRuns;
    this.systemProperties = systemProperties;
    this.enableXmlValidations = enableXmlValidations;
  }

  public Set<SuiteRun> getSuiteRuns() {
    return suiteRuns;
  }

  public boolean isEnableXmlValidations() {
    return enableXmlValidations;
  }

  public Map<String, Object> getSystemProperties() {
    return systemProperties;
  }

  public static Builder builder() {
    return new Builder();
  }

  /**
   * Builds a {@link SuiteDeployment}
   *
   * @author Mulesoft Inc.
   * @since 2.2.0
   */
  public static class Builder {

    private Set<SuiteRun> suiteRuns;
    private Map<String, Object> systemProperties = new HashMap<>();
    private boolean enableXmlValidations = true;

    private Builder() {}

    public Builder withSuiteRuns(Set<SuiteRun> suiteRuns) {
      checkArgument(suiteRuns != null, "Suite runs cannot be null");
      checkArgument(!suiteRuns.isEmpty(), "Suite runs cannot be empty");
      this.suiteRuns = suiteRuns;
      return this;
    }

    public Builder withSystemProperties(Map<String, Object> systemProperties) {
      checkArgument(systemProperties != null, "Parameterization name cannot be null");
      this.systemProperties = systemProperties;
      return this;
    }

    public Builder withEnableXmlValidations(boolean enableXmlValidations) {
      this.enableXmlValidations = enableXmlValidations;
      return this;
    }

    public SuiteDeployment build() {
      return new SuiteDeployment(suiteRuns, systemProperties, enableXmlValidations);
    }
  }

}
