/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.remote.coverage.printer;

import static org.mule.munit.common.util.Preconditions.checkNotNull;
import static java.nio.charset.StandardCharsets.UTF_8;
import org.mule.munit.remote.coverage.model.ApplicationCoverageReport;
import org.mule.munit.remote.coverage.model.CoverageLimits;
import org.mule.munit.remote.coverage.model.MuleFlow;
import org.mule.munit.remote.coverage.model.MuleLocation;
import org.mule.munit.remote.coverage.model.MuleResource;
import org.mule.munit.remote.coverage.printer.model.sonar.SonarCoverage;
import org.mule.munit.remote.coverage.printer.model.sonar.SonarFile;
import org.mule.munit.remote.coverage.printer.model.sonar.SonarLineToCover;

import com.thoughtworks.xstream.XStream;

import java.io.File;
import java.io.FileOutputStream;
import java.io.OutputStreamWriter;
import java.io.Writer;
import java.util.List;
import java.util.stream.Collectors;

/**
 * <p>
 * Generates Sonar Generic Test Data Coverage Report.
 * </p>
 *
 * @author Mulesoft Inc.
 * @since 2.3.0
 */
public class SonarPrinter implements CoverageReportPrinter {

  public static final String PRINTER_NAME = "SONAR";
  private static final String MULESOFT_FLOWS = "src/main/mule";

  private final File reportFile;
  private CoverageLimits coverageLimits;

  public SonarPrinter(CoverageLimits coverageLimits, File reportDirectory) {
    checkNotNull(coverageLimits, "Coverage limits cannot be null");
    checkNotNull(reportDirectory, "Report directory cannot be null");
    this.coverageLimits = coverageLimits;
    this.reportFile = reportDirectory;
  }

  @Override
  public String getPrinterName() {
    return PRINTER_NAME;
  }

  @Override
  public void printReport(ApplicationCoverageReport report) {
    try {
      XStream xStream = new XStream();
      xStream.autodetectAnnotations(true);

      SonarCoverage sonarCoverage = new SonarCoverage();

      for (MuleResource mr : report.getResources()) {
        SonarFile sonarFile = new SonarFile(new File(MULESOFT_FLOWS, mr.getName()).getPath());
        sonarCoverage.addFile(sonarFile);

        for (MuleFlow flow : mr.getFlows()) {
          for (MuleLocation location : flow.getCoveredLocations()) {
            SonarLineToCover sonarLineToCover = new SonarLineToCover(location.getLineNumber(), true);
            sonarFile.addLineToCover(sonarLineToCover);
          }

          List<Integer> coveredLocations =
              flow.getCoveredLocations().stream().map(x -> x.getLineNumber()).collect(Collectors.toList());
          for (MuleLocation location : flow.getLocations()) {
            if (!coveredLocations.contains(location.getLineNumber())) {
              SonarLineToCover sonarLineToCover = new SonarLineToCover(location.getLineNumber(), false);
              sonarFile.addLineToCover(sonarLineToCover);
            }
          }
        }
      }

      Writer writer = new OutputStreamWriter(new FileOutputStream(reportFile), UTF_8);
      writer.write(xStream.toXML(sonarCoverage));
      writer.close();
    } catch (Exception e) {
      e.printStackTrace();
    }
  }

}
