/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.remote.coverage.server;

import java.util.Locale;
import static java.util.Optional.empty;
import static java.util.Optional.of;
import org.apache.commons.lang3.StringUtils;
import org.mule.munit.common.util.Preconditions;

import java.util.Optional;

/**
 * Utility class to parse the string representation of an identifier with the syntax "namespace:name".
 *
 * @author Mulesoft Inc.
 * @since 1.5.0
 *
 */
public final class IdentifierParsingUtils {

  private static final String SEPARATOR = ":";

  private IdentifierParsingUtils() {}

  /**
   * Creates a {@link ComponentIdentifier} from a string representation where the expected format is namespace:name. If the string
   * doesn't contain the namespace then it just needs to be the name of the component and the namespace will default to the passed
   * {@code defaultNamespace}.
   *
   * @param stringRepresentation the component identifier represented as a string
   * @param defaultNamespace     if the passed string representation doesn't have a namespace, the resulting
   *                             {@link ComponentIdentifier} will use this value as the namespace.
   * @return the {@link ComponentIdentifier} created from its string representation.
   */
  public static ComponentIdentifier parseComponentIdentifier(String stringRepresentation, String defaultNamespace) {
    return parseIdentifier(stringRepresentation, defaultNamespace, false);
  }

  private static ComponentIdentifier parseIdentifier(String stringRepresentation, String defaultNamespace, boolean toUpperCase) {
    Preconditions.checkArgument(!StringUtils.isEmpty(stringRepresentation), "identifier cannot be an empty string or null");
    String[] values = stringRepresentation.split(SEPARATOR);

    String namespace;
    String identifier;
    if (values.length == 2) {
      namespace = values[0];
      identifier = values[1];
    } else {
      namespace = defaultNamespace;
      identifier = values[0];
    }

    if (toUpperCase) {
      namespace = namespace.toUpperCase(Locale.getDefault());
      identifier = identifier.toUpperCase(Locale.getDefault());
    }

    return ComponentIdentifier.builder().namespace(namespace).name(identifier).build();
  }

}
