/*
 * Copyright (c) 2015 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.runner;


import org.mule.api.MuleContext;
import org.mule.api.MuleException;
import org.mule.api.config.ConfigurationBuilder;
import org.mule.api.config.MuleProperties;
import org.mule.api.lifecycle.InitialisationException;
import org.mule.modules.interceptor.connectors.ConnectorMethodInterceptorFactory;
import org.mule.munit.common.endpoint.MockEndpointManager;
import org.mule.munit.common.endpoint.MunitSpringFactoryPostProcessor;
import org.mule.munit.common.extensions.MunitPlugin;
import org.mule.munit.common.processor.interceptor.MunitMessageProcessorInterceptorFactory;
import org.mule.munit.runner.mule.context.MunitDomParser;
import org.mule.munit.runner.output.DefaultOutputHandler;
import org.mule.munit.runner.spring.config.MunitSpringXmlConfigurationBuilder;
import org.mule.munit.runner.spring.config.model.MockingConfiguration;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Properties;

import static org.mule.munit.runner.spring.config.MunitSpringXmlConfigurationBuilder.ConfigurationBuilderBuilder;


/**
 * <p>Starts and stops mule</p>
 *
 * @author Mulesoft Inc.
 * @since 3.3.2
 */

public class MuleContextManager {

    private Collection<MunitPlugin> plugins;
    private MockingConfiguration configuration;

    public MuleContextManager(MockingConfiguration configuration) {
        this.configuration = configuration;
    }

    public MuleContext startMule(String resources) throws Exception {
        MuleContext context = createMule(resources);
        return startMule(context);
    }

    public MuleContext startMule(MuleContext context) throws MuleException {
        context.start();
        startPlugins();

        return context;
    }

    public void killMule(MuleContext muleContext) {
        try {
            if (muleContext != null && !muleContext.isStopped()) {
                muleContext.stop();
                stopPlugins();
            }
        } catch (Throwable e1) {

        }
        if (muleContext != null && !muleContext.isDisposed()) {
            muleContext.dispose();
            disposePlugins();
        }

        clearLogginConfiguration();
    }

    public MuleContext createMule(String resources) throws Exception {
        defineLogOutput(resources);

        ConfigurationBuilder configurationBuilder = createConfigurationBuilder(resources);
        List<ConfigurationBuilder> builders = new ArrayList<ConfigurationBuilder>();
        builders.add(configurationBuilder);

        MunitMuleContextFactory contextCreator = new MunitMuleContextFactory(getStartUpProperties(), builders);
        MuleContext context = contextCreator.createMuleContext();

        plugins = new MunitPluginFactory().loadPlugins(context);
        initialisePlugins();

        return context;
    }


    protected ConfigurationBuilder createConfigurationBuilder(String resources) throws Exception {

        ConfigurationBuilderBuilder builder = new ConfigurationBuilderBuilder(resources);

        builder.withMockingConfiguration(configuration)
                .withMunitFactoryPostProcessor(MunitSpringFactoryPostProcessor.MUNIT_FACTORY_POST_PROCESSOR_ID, MunitSpringFactoryPostProcessor.class)
                .withEndpointFactoryClass(MockEndpointManager.class)
                .withBeanToRegister(MunitMessageProcessorInterceptorFactory.ID, MunitMessageProcessorInterceptorFactory.class)
                .withBeanToRegister(ConnectorMethodInterceptorFactory.ID, ConnectorMethodInterceptorFactory.class)
                .withMunitDomParser(new MunitDomParser());

        MunitSpringXmlConfigurationBuilder configuratioBuilder = builder.build();
        return configuratioBuilder;
    }

    /**
     * @since 3.6.x
     */
    private void clearLogginConfiguration() {
        MunitMuleContextFactory.clearLoggingConfiguration();
    }

    private Properties getStartUpProperties() {
        Properties properties = configuration == null ? null : configuration.getStartUpProperties();
        if (properties == null) {
            properties = new Properties();
        }
        if (properties.get(MuleProperties.APP_HOME_DIRECTORY_PROPERTY) == null) {
            properties.setProperty(MuleProperties.APP_HOME_DIRECTORY_PROPERTY, new File(getClass().getResource("/").getPath()).getAbsolutePath());
        }
        return properties;
    }

    private void defineLogOutput(String resources) throws IOException {
        String path = System.getProperty(DefaultOutputHandler.OUTPUT_FOLDER_PROPERTY);
        if (path != null) {
            String name = resources.replace(".xml", "");
            MunitLoggerConfigurer.configureFileLogger(path, name);

        }
    }


    private void startPlugins() throws MuleException {
        for (MunitPlugin plugin : plugins) {
            plugin.start();
        }
    }

    private void disposePlugins() {
        for (MunitPlugin plugin : plugins) {
            plugin.dispose();
        }
    }

    private void stopPlugins() throws MuleException {
        for (MunitPlugin plugin : plugins) {
            plugin.stop();
        }
    }

    private void initialisePlugins() throws InitialisationException {
        for (MunitPlugin plugin : plugins) {
            plugin.initialise();
        }
    }

}
