/*
 * Copyright (c) 2015 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.runner.remote;


import org.apache.commons.lang.StringUtils;
import org.mule.munit.runner.mule.MunitSuiteRunner;
import org.mule.munit.runner.mule.MunitTest;
import org.mule.munit.runner.mule.result.notification.Notification;

import java.io.IOException;
import java.io.ObjectOutputStream;
import java.net.Socket;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

public class MunitRemoteRunner {
    public static final String PORT_PARAMETER = "-port";
    public static final String TEST_PATH_PARAMETER = "-path";
    public static final String RESOURCE_PARAMETER = "-resource";
    public static final String TEST_NAME_PARAMETER = "-test_name";

    public static final String TEST_NAME_TOKEN_SEPARATOR = "<";

    protected String message;
    protected Socket requestSocket;
    protected ObjectOutputStream out;

    public static void main(String args[]) {
        int port = -1;
        String path = null;
        String resource = null;
        String testName = null;


        for (int i = 0; i < args.length; i++) {
            if (args[i].equalsIgnoreCase(RESOURCE_PARAMETER)) {
                resource = args[i + 1];
            }
            if (args[i].equalsIgnoreCase(PORT_PARAMETER)) {
                port = Integer.valueOf(args[i + 1]);
            }
            if (args[i].equalsIgnoreCase(TEST_PATH_PARAMETER)) {
                path = args[i + 1];
            }
            if (args[i].equalsIgnoreCase(TEST_NAME_PARAMETER)) {
                testName = args[i + 1];
            }
        }

        MunitRemoteRunner serverRemoteRunner = new MunitRemoteRunner();
        serverRemoteRunner.run(port, path, resource, testName);

    }

    /**
     * This method's goal is to run a single test suite file and notify events to a listening server.
     *
     * @param port      the port to connect to notify run status
     * @param path      the path of the test suite file being run
     * @param resources a comma separated list of resource files path to be loaded
     * @param testName  the test name, if any, in the event you wish to run a single test. It's a regex.
     */
    public void run(int port, String path, String resources, String testName) {
        RemoteRunnerNotificationListener listener = null;
        try {
            //1. creating a socket to connect to the server
            requestSocket = new Socket("localhost", port);
            System.out.println("[" + this.getClass().getName() + "]" + "Connected to localhost in port " + port);
            out = new ObjectOutputStream(requestSocket.getOutputStream());
            out.flush();

            listener = new RemoteRunnerNotificationListener(out);

            List<String> testNameList = buildTestNamesList(testName);
            MunitSuiteRunner runner;
            try {
                runner = new MunitSuiteRunner(resources, testNameList);
                runner.setNotificationListener(listener);
            } catch (RuntimeException e) {
                listener.notifyRuntimeStartFailure(new Notification(e.getMessage(), MunitTest.stack2string(e)));
                throw e;
            }

            listener.notifyNumberOfTest(runner.getNumberOfTests());

            runner.run();
        } catch (IOException ioException) {
            // TODO: catch other exceptions and notify errors
            ioException.printStackTrace();
        } finally {
            if (listener != null) {
                listener.notifyTestRunEnd(path);
            }

            try {
                //4: Closing connection
                if (null != out) {
                    out.close();
                }
                if (null != requestSocket) {
                    requestSocket.close();
                }
            } catch (IOException ioException) {
                ioException.printStackTrace();
            }
            System.out.println("[" + this.getClass().getName() + "]" + "Done");
        }
        System.exit(0);
    }

    private List<String> buildTestNamesList(String testNames) {
        List<String> testNameList;
        if (StringUtils.isNotBlank(testNames)) {
            testNameList = Arrays.asList(testNames.split(TEST_NAME_TOKEN_SEPARATOR));
        } else {
            testNameList = new ArrayList();
        }
        return testNameList;
    }


}
