/*
 * Copyright (c) 2015 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.runner;


import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.mule.api.MuleContext;
import org.mule.api.MuleException;
import org.mule.api.config.ConfigurationBuilder;
import org.mule.api.config.MuleProperties;
import org.mule.api.lifecycle.InitialisationException;
import org.mule.modules.interceptor.connectors.ConnectorMethodInterceptorFactory;
import org.mule.munit.common.endpoint.MockEndpointManager;
import org.mule.munit.common.endpoint.MunitSpringFactoryPostProcessor;
import org.mule.munit.common.extensions.MunitPlugin;
import org.mule.munit.common.processor.interceptor.MunitMessageProcessorInterceptorFactory;
import org.mule.munit.runner.exception.ExceptionStrategyReplacer;
import org.mule.munit.runner.mule.context.MunitDomParser;
import org.mule.munit.runner.output.DefaultOutputHandler;
import org.mule.munit.runner.properties.ApplicationPropertyLoader;
import org.mule.munit.runner.properties.MUnitUserPropertiesManager;
import org.mule.munit.runner.spring.config.MunitSpringXmlConfigurationBuilder;
import org.mule.munit.runner.spring.config.model.MockingConfiguration;
import org.mule.munit.runner.spring.config.reader.MunitHandlerWrapper;
import org.mule.util.ClassUtils;

import java.io.File;
import java.io.IOException;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.net.URL;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Properties;

import static org.mule.munit.runner.spring.config.MunitSpringXmlConfigurationBuilder.ConfigurationBuilderBuilder;


/**
 * <p>Starts and stops mule</p>
 *
 * @author Mulesoft Inc.
 * @since 3.3.2
 */

public class MuleContextManager {
    private transient Log logger = LogFactory.getLog(this.getClass());

    private Collection<MunitPlugin> plugins;
    private MockingConfiguration configuration;

    public MuleContextManager(MockingConfiguration configuration) {
        this.configuration = configuration;
    }

    public MuleContext startMule(String resources) throws Exception {
        MuleContext context = createMule(resources);
        return startMule(context);
    }

    public MuleContext startMule(MuleContext context) throws MuleException {
        logger.debug("Starting Mule Context tuned by MUnit...");
        context.start();
        startPlugins();

        logger.debug("Mule Context tuned by MUnit Started");
        return context;
    }

    public void killMule(MuleContext muleContext) {
        logger.debug("Stopping Mule Context tuned by MUnit...");
        try {
            if (muleContext != null && !muleContext.isStopped()) {
                muleContext.stop();
                stopPlugins();
            }
        } catch (Throwable e1) {

        }
        if (muleContext != null && !muleContext.isDisposed()) {
            muleContext.dispose();
            disposePlugins();
        }

        logger.debug("Mule Context stopped");
        clearLogginConfiguration();
    }

    public MuleContext createMule(String resources) throws Exception {
        logger.debug("Creating Mule Context tuned by MUnit...");

        defineLogOutput(resources);

        defineBeanConstructorArgLimit();

        loadMuleAppProperties();

        ConfigurationBuilder configurationBuilder = createConfigurationBuilder(resources);
        List<ConfigurationBuilder> builders = new ArrayList<ConfigurationBuilder>();
        builders.add(configurationBuilder);

        MunitMuleContextFactory contextCreator = new MunitMuleContextFactory(getStartUpProperties(), builders);
        MuleContext context = contextCreator.createMuleContext();

        replaceExceptionStrategies(context);

        plugins = new MunitPluginFactory().loadPlugins(context);
        initialisePlugins();

        return context;
    }

    private static final Integer CONSTRUCTOR_ARG_LIMIT = 13;

    protected void defineBeanConstructorArgLimit() {
        try {
            Method method = MunitHandlerWrapper.class.getDeclaredMethod("setConstructorArgLimit", Integer.class);
            if (null != method) {
                method.invoke(null, CONSTRUCTOR_ARG_LIMIT);
            }
        } catch (NoSuchMethodException e) {
            logger.debug("Using MUnit Support that doesn't support constructor parameter definition");
        } catch (InvocationTargetException e) {
            logger.debug("Fail to set constructor arg limit in MUnit support");
        } catch (IllegalAccessException e) {
            logger.debug("Fail to set constructor arg limit in MUnit support");
        }
    }

    protected ConfigurationBuilder createConfigurationBuilder(String resources) throws Exception {
        logger.debug("Creating ConfigurationBuilder for resources: " + resources);

        ConfigurationBuilderBuilder builder = new ConfigurationBuilderBuilder(resources);

        builder.withMockingConfiguration(configuration)
                .withMunitFactoryPostProcessor(MunitSpringFactoryPostProcessor.MUNIT_FACTORY_POST_PROCESSOR_ID, MunitSpringFactoryPostProcessor.class)
                .withEndpointFactoryClass(MockEndpointManager.class)
                .withBeanToRegister(MunitMessageProcessorInterceptorFactory.ID, MunitMessageProcessorInterceptorFactory.class)
                .withBeanToRegister(ConnectorMethodInterceptorFactory.ID, ConnectorMethodInterceptorFactory.class)
                .withBeanToRegister(ExceptionStrategyReplacer.ID, ExceptionStrategyReplacer.class)
                .withMunitDomParser(new MunitDomParser());

        MunitSpringXmlConfigurationBuilder configuratioBuilder = builder.build();
        return configuratioBuilder;
    }

    private void replaceExceptionStrategies(MuleContext context) {
        logger.debug("Replacing exception strategies with MUnit proxies...");
        ExceptionStrategyReplacer replacer = context.getRegistry().get(ExceptionStrategyReplacer.ID);
        replacer.setMuleContext(context);
        replacer.replace();
    }

    /**
     * @since 3.6.x
     */
    private void clearLogginConfiguration() {
        MunitMuleContextFactory.clearLoggingConfiguration();
    }

    private Properties getStartUpProperties() {
        logger.debug("Loading startup properties...");
        Properties properties = configuration == null ? null : configuration.getStartUpProperties();
        if (properties == null) {
            properties = new Properties();
        }
        if (properties.get(MuleProperties.APP_HOME_DIRECTORY_PROPERTY) == null) {
            properties.setProperty(MuleProperties.APP_HOME_DIRECTORY_PROPERTY, new File(getClass().getResource("/").getPath()).getAbsolutePath());
        }
        logger.debug("Startup properties loaded: [" + properties.toString() + "]");
        return properties;
    }

    private void defineLogOutput(String resources) throws IOException {
        String path = System.getProperty(DefaultOutputHandler.OUTPUT_FOLDER_PROPERTY);
        if (path != null) {
            String name = resources.replace(".xml", "");
            MunitLoggerConfigurer.configureFileLogger(path, name);

        }
    }


    private void startPlugins() throws MuleException {
        logger.debug("Starting MUnit plugins...");
        for (MunitPlugin plugin : plugins) {
            plugin.start();
            logger.debug(plugin.getClass().getName() + " plugin started");
        }
    }

    private void disposePlugins() {
        logger.debug("Disposing MUnit plugins...");
        for (MunitPlugin plugin : plugins) {
            plugin.dispose();
            logger.debug(plugin.getClass().getName() + " plugin disposed");
        }
    }

    private void stopPlugins() throws MuleException {
        for (MunitPlugin plugin : plugins) {
            plugin.stop();
        }
    }

    private void initialisePlugins() throws InitialisationException {
        logger.debug("Initializing MUnit plugins...");
        for (MunitPlugin plugin : plugins) {
            plugin.initialise();
            logger.debug(plugin.getClass().getName() + " plugin initialised");
        }
    }

    private void loadMuleAppProperties() {
        logger.info("Loading application properties...");;
        ApplicationPropertyLoader propertyLoader = new ApplicationPropertyLoader(new MUnitUserPropertiesManager(),logger);
        URL url = ClassUtils.getResource(ApplicationPropertyLoader.DEFAULT_APP_PROPERTIES_RESOURCE, getClass());
        propertyLoader.loadAndSetApplicationProperties(url);
        logger.debug("Application properties loading [DONE]");
    }

}
