/*
 * Copyright (c) 2015 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.runner.domain;

import org.apache.commons.lang.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.mule.util.ClassUtils;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.Map;
import java.util.Properties;

/**
 * It knows how to load and read the file mule-deploy.properties from the classpath.
 * As long as the file belongs to the current app and/or the related domain.
 */

public class MuleDeployPropertyLoader {
    public static final String DEFAULT_MULE_DEPLOY_PROPERTIES_RESOURCE = "mule-deploy.properties";

    private static final String MULE_DEPLOY_PROPERTIES_RELATIVE_PATH = File.separator + "target" + File.separator + "classes" + File.separator + MuleDeployPropertyLoader.DEFAULT_MULE_DEPLOY_PROPERTIES_RESOURCE;
    private static final String MULE_DEPLOY_PROPERTIES_TEST_RELATIVE_PATH = File.separator + "target" + File.separator + "test-classes" + File.separator + MuleDeployPropertyLoader.DEFAULT_MULE_DEPLOY_PROPERTIES_RESOURCE;

    private transient Log log = LogFactory.getLog(this.getClass());;

    private final String projectName;

    private Map<String, String> domainDeployPropertiesMap = null;
    private Map<String, String> applicationDeployPropertiesMap = null;

    public MuleDeployPropertyLoader(String projectName) {
        this.projectName = projectName;
    }

    public Map<String, String> getApplicationDeployProperties() {
        if (null == applicationDeployPropertiesMap) {
            loadApplicationDeployProperties();
        }
        return applicationDeployPropertiesMap;
    }

    public Map<String, String> getDomainDeployProperties() {
        if (null == applicationDeployPropertiesMap) {
            loadApplicationDeployProperties();
        }

        if (null == domainDeployPropertiesMap) {
            String domainName = applicationDeployPropertiesMap.get("domain");
            loadDomainDeployProperties(domainName);
        }

        return domainDeployPropertiesMap;
    }

    private void loadApplicationDeployProperties() {

        applicationDeployPropertiesMap = new HashMap<String, String>();
        URL url = null;
        Enumeration<URL> urls = ClassUtils.getResources(DEFAULT_MULE_DEPLOY_PROPERTIES_RESOURCE, getClass());
        while (urls.hasMoreElements()) {
            URL u = urls.nextElement();
            if (doesTheFileBelongToTheApplication(u)) {
                url = u;
            }
        }

        if (null != url) {
            log.debug("Loaded " + DEFAULT_MULE_DEPLOY_PROPERTIES_RESOURCE + " file from:" + url.getPath().toString());
            loadDeployProperties(url, applicationDeployPropertiesMap);
        }
    }

    private void loadDomainDeployProperties(String domainName) {
        domainDeployPropertiesMap = new HashMap<String, String>();

        if (StringUtils.isBlank(domainName)) {
            log.debug("Attempting to load " + DEFAULT_MULE_DEPLOY_PROPERTIES_RESOURCE + " for a blank domain will return not properties");
        }

        URL url = null;
        Enumeration<URL> urls = ClassUtils.getResources(DEFAULT_MULE_DEPLOY_PROPERTIES_RESOURCE, getClass());
        while (urls.hasMoreElements()) {
            URL u = urls.nextElement();
            if (doesTheFileBelongToTheDomain(u, domainName)) {
                url = u;
            }
        }

        if (null != url) {
            log.debug("Loaded " + DEFAULT_MULE_DEPLOY_PROPERTIES_RESOURCE + " file from:" + url.getPath().toString());
            loadDeployProperties(url, domainDeployPropertiesMap);
        }
    }

    private void loadDeployProperties(URL resourceUrl, Map<String, String> propertiesMap) {
        if (null == resourceUrl) {
            log.warn(DEFAULT_MULE_DEPLOY_PROPERTIES_RESOURCE + " file was not found");
            return;
        }
        try {
            propertiesMap.putAll(loadProperties(resourceUrl));
        } catch (IOException e) {
            log.warn(DEFAULT_MULE_DEPLOY_PROPERTIES_RESOURCE + " could not be loaded.");
        }
    }

    private boolean doesTheFileBelongToTheApplication(URL url) {
        if (url.getPath().toString().contains(projectName + MULE_DEPLOY_PROPERTIES_RELATIVE_PATH) ||
                url.getPath().toString().contains(projectName + MULE_DEPLOY_PROPERTIES_TEST_RELATIVE_PATH)) {
            return true;
        }
        return false;
    }

    private boolean doesTheFileBelongToTheDomain(URL url, String domainName) {
        String domainMuleDeployPropertiesRelativePath = File.separator + domainName + File.separator;

        if (url.getPath().toString().contains(domainMuleDeployPropertiesRelativePath)) {
            return true;
        }
        return false;
    }

    private Map<String, String> loadProperties(URL propsFile) throws IOException {
        Map<String, String> appPropsMap = new HashMap<String, String>();
        Properties props = loadPropertiesFromFile(propsFile);
        for (Object key : props.keySet()) {
            appPropsMap.put(key.toString(), props.getProperty(key.toString()));
        }
        return appPropsMap;
    }

    private Properties loadPropertiesFromFile(URL url) throws IOException {
        Properties props;
        if (url == null) {
            throw new IOException("Invalid file URL!");
        }
        InputStream is = url.openStream();
        try {
            props = new Properties();
            props.load(is);
        } finally {
            is.close();
        }
        return props;
    }

}
