/*
 * Copyright (c) 2015 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.runner.domain;

import org.apache.commons.lang.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.mule.api.MuleContext;

/**
 * This class knows how to create a Domain context based on the application and and it's classpath.
 * <p/>
 * Based on the properties loaded it will decide whether or not to create a Domain context.
 *
 * @author Mulesoft Inc.
 * @since 1.2.0
 */
public class MunitDomainContextBuilder {
    private transient Log logger = LogFactory.getLog(this.getClass());

    private static final String DOMAIN_CONFIG_FILE_NAME = "mule-domain-config.xml";

    private final String projectName;
    private DomainContextBuilder domainContextBuilder;
    private MuleDeployPropertyLoader muleDeployPropertyLoader;

    public MunitDomainContextBuilder(String projectName) {
        this.projectName = projectName;
        this.domainContextBuilder = new DomainContextBuilder();
        this.muleDeployPropertyLoader = new MuleDeployPropertyLoader(projectName);
    }

    /**
     * It builds a domain context based on the current classpath and properties being loaded.
     *
     * @return null, if the domain if the properties files can not be loaded, the domain is default, or if the domain resources are empty
     * @throws Exception
     */
    public MuleContext buildDomainContextIfRequired() throws Exception {
        MuleContext domainContext = null;

        if (StringUtils.isBlank(projectName)) {
            logger.debug("No project name provided Domain Context won't be built");
            return domainContext;
        }

        String domainResources = loadDomainResources();
        if (StringUtils.isNotBlank(domainResources)) {
            logger.debug("About to build Domain Context with resources: " + domainResources);
            domainContext = domainContextBuilder.setDomainConfig(domainResources).build();
            logger.debug("Domain Context built");
        } else {
            logger.debug("No domain resources found, Domain Context won't be built");
        }

        return domainContext;
    }

    private String loadDomainResources() {
        String domainResources = "";

        logger.info("Loading mule-deploy.properties ...");

        String domainName = muleDeployPropertyLoader.getApplicationDeployProperties().get("domain");
        if (StringUtils.isNotBlank(domainName) && !"default".equals(domainName.toLowerCase())) {
            domainName = domainName.toLowerCase();
            logger.debug("Domain found: " + domainName);
            domainResources = DOMAIN_CONFIG_FILE_NAME;
        } else {
            logger.debug("No domain or default domain found.");
        }

        return domainResources;
    }

    public void setDomainContextBuilder(DomainContextBuilder domainContextBuilder) {
        this.domainContextBuilder = domainContextBuilder;
    }

    public void setMuleDeployPropertyLoader(MuleDeployPropertyLoader muleDeployPropertyLoader) {
        this.muleDeployPropertyLoader = muleDeployPropertyLoader;
    }
}