/*
 * Copyright (c) 2015 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.runner;


import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.mule.api.MuleContext;
import org.mule.api.MuleException;
import org.mule.api.config.ConfigurationBuilder;
import org.mule.api.config.MuleProperties;
import org.mule.api.lifecycle.InitialisationException;
import org.mule.api.registry.RegistrationException;
import org.mule.modules.interceptor.connectors.ConnectorMethodInterceptorFactory;
import org.mule.munit.common.endpoint.MockEndpointManager;
import org.mule.munit.common.endpoint.MunitSpringFactoryPostProcessor;
import org.mule.munit.common.extensions.MunitPlugin;
import org.mule.munit.common.processor.interceptor.MunitMessageProcessorInterceptorFactory;
import org.mule.munit.runner.domain.MunitDomainContextBuilder;
import org.mule.munit.runner.exception.ExceptionStrategyReplacer;
import org.mule.munit.runner.mule.context.MunitDomParser;
import org.mule.munit.runner.properties.ApplicationPropertyLoader;
import org.mule.munit.runner.properties.MUnitUserPropertiesManager;
import org.mule.munit.runner.spring.config.MunitSpringXmlConfigurationBuilder;
import org.mule.munit.runner.spring.config.model.MockingConfiguration;
import org.mule.munit.runner.spring.config.reader.MunitHandlerWrapper;
import org.mule.util.ClassUtils;

import java.io.File;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.net.URL;
import java.util.*;

import static org.mule.munit.runner.spring.config.MunitSpringXmlConfigurationBuilder.ConfigurationBuilderBuilder;


/**
 * <p>Starts and stops mule</p>
 *
 * @author Mulesoft Inc.
 * @since 3.3.2
 */

public class MuleContextManager {
    private static final Integer CONSTRUCTOR_ARG_LIMIT = 13;
    public static final String USE_XALAN_TRANSFORMER_PROPERTY = "useXalanTransformer";

    private transient Log logger = LogFactory.getLog(this.getClass());

    private MUnitUserPropertiesManager propertiesManager = new MUnitUserPropertiesManager();

    private Collection<MunitPlugin> plugins;
    private MockingConfiguration configuration;

    private Map<MuleContext, MuleContext> appDomainMap = new HashMap<MuleContext, MuleContext>();

    public MuleContextManager(MockingConfiguration configuration) {
        this.configuration = configuration;
    }

    public MuleContext startMule(String resources, String projectName) throws Exception {
        MuleContext context = createMule(resources, projectName);
        return startMule(context);
    }

    public MuleContext startMule(MuleContext context) throws MuleException {
        logger.debug("Starting Mule Context tuned by MUnit...");

        context.start();

        startPlugins();

        logger.debug("Mule Context tuned by MUnit Started");

        return context;
    }

    public void killMule(MuleContext muleContext) {
        logger.debug("Shooting down Mule Context tuned by MUnit...");

        stopMuleContext(muleContext);

        disposeMuleContext(muleContext);

        logger.debug("Mule Context shot down");

        clearLogginConfiguration();
    }

    public MuleContext createMule(String resources, String projectName) throws Exception {
        logger.debug("Creating Mule Context tuned by MUnit...");

        defineBeanConstructorArgLimit();

        loadMuleAppProperties();
        loadAdditionalSystemProperties();

        ConfigurationBuilder configurationBuilder = createConfigurationBuilder(resources, projectName);

        MuleContext domainContext = new MunitDomainContextBuilder(projectName).buildDomainContextIfRequired();
        if (null != domainContext) {
            ((MunitSpringXmlConfigurationBuilder) configurationBuilder).setDomainContext(domainContext);
        }

        List<ConfigurationBuilder> builders = new ArrayList<ConfigurationBuilder>();
        builders.add(configurationBuilder);

        MunitMuleContextFactory contextCreator = new MunitMuleContextFactory(getStartUpProperties(), builders);
        MuleContext context = contextCreator.createMuleContext();

        loadApplicationPropertiesToMuleContext(context);

        appDomainMap.put(context, domainContext);

        replaceExceptionStrategies(context);

        plugins = new MunitPluginFactory().loadPlugins(context);
        initialisePlugins();

        return context;
    }



    protected void defineBeanConstructorArgLimit() {
        try {
            Method method = MunitHandlerWrapper.class.getDeclaredMethod("setConstructorArgLimit", Integer.class);
            if (null != method) {
                method.invoke(null, CONSTRUCTOR_ARG_LIMIT);
            }
        } catch (NoSuchMethodException e) {
            logger.debug("Using MUnit Support that doesn't support constructor parameter definition");
        } catch (InvocationTargetException e) {
            logger.debug("Fail to set constructor arg limit in MUnit support");
        } catch (IllegalAccessException e) {
            logger.debug("Fail to set constructor arg limit in MUnit support");
        }
    }

    protected ConfigurationBuilder createConfigurationBuilder(String resources, String projectName) throws Exception {
        logger.debug("Creating ConfigurationBuilder for resources: " + resources);

        ConfigurationBuilderBuilder builder = new ConfigurationBuilderBuilder(resources);

        builder.withMockingConfiguration(configuration)
                .withMunitFactoryPostProcessor(MunitSpringFactoryPostProcessor.MUNIT_FACTORY_POST_PROCESSOR_ID, MunitSpringFactoryPostProcessor.class)
                .withEndpointFactoryClass(MockEndpointManager.class)
                .withBeanToRegister(MunitMessageProcessorInterceptorFactory.ID, MunitMessageProcessorInterceptorFactory.class)
                .withBeanToRegister(ConnectorMethodInterceptorFactory.ID, ConnectorMethodInterceptorFactory.class)
                .withBeanToRegister(ExceptionStrategyReplacer.ID, ExceptionStrategyReplacer.class)
                .withMunitDomParser(new MunitDomParser());

        MunitSpringXmlConfigurationBuilder configurationBuilder = builder.build();

        return configurationBuilder;
    }

    private void replaceExceptionStrategies(MuleContext context) {
        logger.debug("Replacing exception strategies with MUnit proxies...");
        ExceptionStrategyReplacer replacer = context.getRegistry().get(ExceptionStrategyReplacer.ID);
        replacer.setMuleContext(context);
        replacer.replace();
    }

    /**
     * @since 3.6.x
     */
    private void clearLogginConfiguration() {
        MunitMuleContextFactory.clearLoggingConfiguration();
    }

    private Properties getStartUpProperties() {
        logger.debug("Loading startup properties...");
        Properties properties = configuration == null ? null : configuration.getStartUpProperties();
        if (properties == null) {
            properties = new Properties();
        }
        if (properties.get(MuleProperties.APP_HOME_DIRECTORY_PROPERTY) == null) {
            properties.setProperty(MuleProperties.APP_HOME_DIRECTORY_PROPERTY, new File(getClass().getResource("/").getPath()).getAbsolutePath());
        }
        logger.debug("Startup properties loaded: [" + properties.toString() + "]");
        return properties;
    }

    private void startPlugins() throws MuleException {
        logger.debug("Starting MUnit plugins...");
        for (MunitPlugin plugin : plugins) {
            plugin.start();
            logger.debug(plugin.getClass().getName() + " plugin started");
        }
    }

    private void disposePlugins() {
        logger.debug("Disposing MUnit plugins...");
        for (MunitPlugin plugin : plugins) {
            plugin.dispose();
            logger.debug(plugin.getClass().getName() + " plugin disposed");
        }
    }

    private void stopPlugins() throws MuleException {
        for (MunitPlugin plugin : plugins) {
            plugin.stop();
        }
    }

    private void initialisePlugins() throws InitialisationException {
        logger.debug("Initializing MUnit plugins...");
        for (MunitPlugin plugin : plugins) {
            plugin.initialise();
            logger.debug(plugin.getClass().getName() + " plugin initialised");
        }
    }

    private void loadMuleAppProperties() {
        logger.info("Loading mule-app.properties ...");
        ApplicationPropertyLoader propertyLoader = new ApplicationPropertyLoader(propertiesManager, logger);
        URL url = ClassUtils.getResource(ApplicationPropertyLoader.DEFAULT_APP_PROPERTIES_RESOURCE, getClass());
        propertyLoader.loadAndSetApplicationProperties(url);
        logger.debug("mule-app.properties loading done");
    }

    private void loadApplicationPropertiesToMuleContext(MuleContext context) {
        logger.info("Loading application properties to Mule Context");
        Map<String, Object> appProp = new HashMap<String, Object>();
        for (Map.Entry<String, String> e : propertiesManager.getApplicationProperties().entrySet()) {
            appProp.put(e.getKey(), (Object) e.getValue());
        }

        try {
            context.getRegistry().registerObjects(appProp);

        } catch (RegistrationException e) {
            logger.warn("There has been an error loading the application properties to the Mule Context", e);
        }
        logger.debug("Loading application properties to Mule Context done");
    }


    /**
     * Fix for MU-537. Set this system property to force usage of xalan connector
     */
    private void loadAdditionalSystemProperties() {
        if("true".equals(System.getProperty(USE_XALAN_TRANSFORMER_PROPERTY))) {
            System.setProperty("javax.xml.transform.TransformerFactory","org.apache.xalan.processor.TransformerFactoryImpl");
        }
    }

    private void stopMuleContext(MuleContext muleContext) {
        logger.debug("Stopping Mule Context tuned by MUnit...");
        try {
            if (muleContext != null && !muleContext.isStopped()) {
                muleContext.stop();
                stopPlugins();

                if (null != appDomainMap.get(muleContext) && !appDomainMap.get(muleContext).isStopped()) {
                    logger.debug("Stopping Mule Domain Context tuned...");
                    appDomainMap.get(muleContext).stop();
                    logger.debug("Mule Domain Context tuned stopped");
                }
            }
        } catch (Throwable e) {
            logger.debug("There has been an error while stopping Mule Context", e);
        }
        logger.debug("Mule Context stopped");
    }

    private void disposeMuleContext(MuleContext muleContext) {
        if (muleContext != null && !muleContext.isDisposed()) {
            logger.debug("Disposing Mule Context tuned by MUnit...");
            muleContext.dispose();
            disposePlugins();

            if (null != appDomainMap.get(muleContext) && !appDomainMap.get(muleContext).isDisposed()) {
                logger.debug("Disposing Mule Domain Context tuned...");

                appDomainMap.get(muleContext).dispose();

                logger.debug("Mule Domain Context tuned disposed");
            }
            logger.debug("Mule Context tuned by MUnit disposed");
        }
    }
}
