/*
 * Copyright (c) 2015 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.runner;

import org.apache.commons.lang.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.mule.api.MuleContext;
import org.mule.api.MuleEvent;
import org.mule.api.MuleException;
import org.mule.api.construct.FlowConstruct;
import org.mule.munit.assertion.processors.MunitAfterSuite;
import org.mule.munit.assertion.processors.MunitBeforeSuite;
import org.mule.munit.assertion.processors.MunitFlow;
import org.mule.munit.runner.exception.BeforeSuiteException;
import org.mule.munit.runner.exception.FlowException;
import org.mule.munit.runner.mule.MunitTest;
import org.mule.munit.runner.mule.result.notification.DummySuiteRunnerEventListener;
import org.mule.munit.runner.mule.result.notification.Notification;
import org.mule.munit.runner.mule.result.notification.SuiteRunnerEventListener;
import org.mule.munit.runner.output.TestOutputHandler;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

/**
 * <p>Runs the Munit Suite</p>
 * <p/>
 * <p>T is the expected result from the suite run</p>
 *
 * @author Mulesoft Inc.
 * @since 3.3.2
 */
public abstract class MunitRunner<T> {
    private transient Log logger = LogFactory.getLog(this.getClass());

    private TestOutputHandler handler;

    private MuleContext muleContext;
    private MuleContextManager muleContextManager;

    private SuiteRunnerEventListener suiteRunnerEventListener = new DummySuiteRunnerEventListener();


    public MunitRunner(TestOutputHandler handler, MuleContextManager muleContextManager, MuleContext muleContext) {
        this.handler = handler;
        this.muleContext = muleContext;
        this.muleContextManager = muleContextManager;
    }

    /**
     * <p>Runs all the tests of the suite</p>
     *
     * @return The suite result
     * @throws Exception If the suite fails
     */
    protected abstract T runSuite() throws Exception;

    /**
     * <p>Get the name of the suite</p>
     *
     * @return The suite name
     */
    protected abstract String getSuiteName();

    /**
     * <p>Runs the suite based on the constructor arguments </p>
     *
     * @return The suite result
     */
    public T run() {
        logger.debug("About to run MUnit suite: " + getSuiteName() + " ...");
        handler.printTestName(getSuiteName());
        try {
            processBeforeSuites();

            T result = runSuite();
            logger.debug("Tests in MUnit suite: " + getSuiteName() + " run");
            return result;

        } catch (BeforeSuiteException e) {
            logger.error("Before suites execution failed", e.getCause());
            throw new RuntimeException("Before suites execution failed", e.getCause());
        } catch (Throwable e) {
            logger.error("Could not Run the suite: " + getSuiteName(), e);
            throw new RuntimeException("Could not Run the suite", e);
        } finally {
            processAfterSuites();
        }

    }

    public void setSuiteRunnerEventListener(SuiteRunnerEventListener suiteRunnerEventListener) {
        this.suiteRunnerEventListener = suiteRunnerEventListener;
    }

    private void processBeforeSuites() {
        try {
            logger.debug("Executing Before Suite scopes...");
            List<MunitFlow> beforeSuites = lookupFlows(MunitBeforeSuite.class);
            if (!beforeSuites.isEmpty()) {
                process(beforeSuites, muleEvent(beforeSuites.get(0)));
            }
        } catch (FlowException flowException) {
            Throwable e = flowException.getCause();
            Notification notification = getNotification(e, String.format("Before suite %s failed", flowException.getFlowName()));
            if (e instanceof AssertionError) {
                suiteRunnerEventListener.notifyBeforeSuiteFailure(notification);
            } else {
                suiteRunnerEventListener.notifyBeforeSuiteError(notification);
            }
            throw new BeforeSuiteException(e);
        }
    }

    private void processAfterSuites() {
        try {
            logger.debug("Executing After Suite scopes...");
            List<MunitFlow> afterSuites = lookupFlows(MunitAfterSuite.class);
            if (!afterSuites.isEmpty()) {
                process(afterSuites, muleEvent(afterSuites.get(0)));
            }
        } catch (FlowException flowException) {
            Throwable e = flowException.getCause();
            logger.error("After suites execution failed", e);
            Notification notification = getNotification(e, String.format("After suite %s failed", flowException.getFlowName()));
            if (e instanceof AssertionError) {
                suiteRunnerEventListener.notifyAfterSuiteFailure(notification);
            } else {
                suiteRunnerEventListener.notifyAfterSuiteError(notification);
            }
            throw new RuntimeException("After suites execution failed", e);
        } finally {
            muleContextManager.killMule(muleContext);
        }
    }

    private MuleEvent muleEvent(FlowConstruct flowConstruct) {
        return MuleEventBuilderWrapper.muleEvent(muleContext, flowConstruct);
    }

    private void process(Collection<MunitFlow> flowConstructs, MuleEvent event) throws FlowException {
        String flowName = StringUtils.EMPTY;
        try {
            for (MunitFlow flowConstruct : flowConstructs) {
                flowName = flowConstruct.getName();
                handler.printDescription(flowName, flowConstruct.getDescription());
                (flowConstruct).process(event);
            }
        } catch (Throwable e) {
            throw new FlowException(flowName, e);
        }
    }

    private List<MunitFlow> lookupFlows(Class<? extends MunitFlow> munitClass) {
        return new ArrayList<MunitFlow>(muleContext.getRegistry()
                .lookupObjects(munitClass));
    }

    private Notification getNotification(Throwable cause, String message) {
        RuntimeException e = new RuntimeException(message, cause);
        return new Notification(e.getMessage(), MunitTest.stack2string(e));
    }
}
