/*
 * Copyright (c) 2015 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.runner.mule;

import org.mule.api.MuleContext;
import org.mule.munit.runner.MuleContextManager;
import org.mule.munit.runner.MunitRunner;
import org.mule.munit.runner.mule.result.SuiteResult;
import org.mule.munit.runner.mule.result.notification.DummySuiteRunnerEventListener;
import org.mule.munit.runner.mule.result.notification.Notification;
import org.mule.munit.runner.mule.result.notification.SuiteRunnerEventListener;
import org.mule.munit.runner.output.DefaultOutputHandler;
import org.mule.munit.runner.output.TestOutputHandler;
import org.mule.munit.runner.spring.config.model.MockingConfiguration;

import java.util.List;


/**
 * <p>
 * The Munit test runner
 * </p>
 *
 * @author Mulesoft Inc.
 * @since 3.3.2
 */
public class SuiteRunner {

    private MuleContext muleContext;
    private MunitSuite suite;
    private TestOutputHandler handler = new DefaultOutputHandler();
    private MuleContextManager muleContextManager;
    private SuiteRunnerEventListener suiteRunnerEventListener = new DummySuiteRunnerEventListener();


    public SuiteRunner(String resources, List<String> testNameList, String projectName) {
        this(resources, testNameList, projectName, null, new DummySuiteRunnerEventListener());
    }

    public SuiteRunner(String resources, List<String> testNameList, String projectName, SuiteRunnerEventListener suiteRunnerEventListener) {
        this(resources, testNameList, projectName, null, suiteRunnerEventListener);
    }

    public SuiteRunner(String resources, List<String> testNameList, String projectName, MockingConfiguration configuration, SuiteRunnerEventListener suiteRunnerEventListener) {
        this.suiteRunnerEventListener = suiteRunnerEventListener;
        try {
            //TODO: dds change this so we can inject the mulecontextmanager
            muleContextManager = new MuleContextManager(configuration);
            muleContext = muleContextManager.startMule(resources, projectName);

            suite = new MunitSuiteBuilder(muleContext, handler).build(resources, testNameList);

        } catch (Throwable e) {
            handleSuiteStartFailure(e);
            throw new RuntimeException(e);
        }
    }

    public SuiteResult run() {
        MunitRunner<SuiteResult> munitRunner = new MunitRunner<SuiteResult>(handler, muleContextManager, muleContext) {

            @Override
            protected SuiteResult runSuite() throws Exception {
                return suite.run();
            }

            @Override
            protected String getSuiteName() {
                return suite.getName();
            }
        };
        munitRunner.setSuiteRunnerEventListener(suiteRunnerEventListener);
        suite.setSuiteRunnerEventListener(suiteRunnerEventListener);

        suiteRunnerEventListener.notifySuiteStart(suite.getName());
        SuiteResult suiteResult = munitRunner.run();
        suiteRunnerEventListener.notifySuiteEnd(suiteResult);
        return suiteResult;
    }

    private void handleSuiteStartFailure(Throwable originalFailure) {
        try {
            suiteRunnerEventListener.notifySuiteStartFailure(new Notification(originalFailure.getMessage(), MunitTest.stack2string(originalFailure)));
            muleContextManager.killMule(muleContext);
        } catch (Throwable e){
            throw new RuntimeException(originalFailure);
        }
    }

    public void setSuiteRunnerEventListener(SuiteRunnerEventListener suiteRunnerEventListener) {
        this.suiteRunnerEventListener = suiteRunnerEventListener;
    }

    public int getNumberOfTests() {
        return suite.getNumberOfTests();
    }

    public void setHandler(TestOutputHandler handler) {
        this.handler = handler;
    }
}
