/*
 * Copyright (c) 2015 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.runner.mule;

import org.apache.commons.lang.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.mule.api.MuleContext;
import org.mule.api.MuleEvent;
import org.mule.api.MuleException;
import org.mule.api.construct.FlowConstruct;
import org.mule.munit.assertion.processors.MunitFlow;
import org.mule.munit.assertion.processors.MunitTestFlow;
import org.mule.munit.common.MunitCore;
import org.mule.munit.common.exception.MunitError;
import org.mule.munit.runner.MuleEventBuilderWrapper;
import org.mule.munit.runner.MunitMuleEventBuilder;
import org.mule.munit.runner.mule.result.TestResult;
import org.mule.munit.runner.mule.result.notification.Notification;
import org.mule.munit.runner.output.TestOutputHandler;

import java.io.PrintWriter;
import java.io.StringWriter;
import java.util.List;

import static junit.framework.Assert.fail;
import static org.mule.munit.common.MunitCore.buildMuleStackTrace;

/**
 * <p>MUnit Test</p>
 *
 * @author Mulesoft Inc.
 * @since 3.3.2
 */
public class MunitTest {
    private transient Log logger = LogFactory.getLog(this.getClass());

    /**
     * <p>The MUnit flows that have to be run before the MUnit test.</p>
     */
    private List<MunitFlow> before;

    /**
     * <p>The MUnit flows that have to be run after the MUnit test.</p>
     */
    private List<MunitFlow> after;

    /**
     * <p>The MUnit test.</p>
     */
    private MunitTestFlow test;

    /**
     * <p>The Output handler to be use.</p>
     */
    private TestOutputHandler outputHandler;
    private MuleContext muleContext;

    public static String stack2string(Throwable e) {
        try {
            StringWriter sw = new StringWriter();
            PrintWriter pw = new PrintWriter(sw);
            e.printStackTrace(pw);
            return sw.toString();
        } catch (Exception e2) {
            return "";
        }
    }

    public MunitTest(List<MunitFlow> before,
                     MunitTestFlow test,
                     List<MunitFlow> after,
                     TestOutputHandler outputHandler, MuleContext muleContext) {
        this.before = before;
        this.after = after;
        this.test = test;
        this.outputHandler = outputHandler;
        this.muleContext = muleContext;
    }

    public String getName() {
        return test.getName();
    }

    public boolean isIgnore() {
        return test.isIgnore();
    }

    public TestResult run() {
        TestResult result = new TestResult(getName());
        logger.debug("About to run MUnit test: " + getName());

        if (test.isIgnore()) {
            logger.debug("MUnit test: " + getName() + " is ignored it won't run.");
            result.setSkipped(true);
            return result;
        }

        long start = System.currentTimeMillis();
        MuleEvent event = muleEvent(test);

        try {
            runBefore(event);
            showDescription();

            test.process(event);

            if (StringUtils.isNotBlank(test.getExpectException())) {
                fail("Exception matching '" + test.getExpectException() + "', but wasn't thrown");
            }
        } catch (final AssertionError t) {
            result.setFailure(buildNotifcationFrom(t));
        } catch (final MuleException e) {
            try {
                if (!test.expectException(e, event)) {

                    Throwable cause = e.getCause();
                    if (cause != null && AssertionError.class.isAssignableFrom(cause.getClass())) {
                        cause.setStackTrace(buildMuleStackTrace(event.getMuleContext())
                                .toArray(new StackTraceElement[]{}));

                        Notification notification = buildNotifcationFrom(cause);
                        result.setFailure(notification);
                    } else {
                        e.setStackTrace(buildMuleStackTrace(event.getMuleContext())
                                .toArray(new StackTraceElement[]{}));

                        Notification notification = buildNotifcationFrom(e);
                        result.setError(notification);
                    }
                }
            } catch (final AssertionError t) {
                t.setStackTrace(buildMuleStackTrace(event.getMuleContext())
                        .toArray(new StackTraceElement[]{}));
                result.setFailure(buildNotifcationFrom(t));
            } catch (final MunitError t) {
                t.setStackTrace(buildMuleStackTrace(event.getMuleContext())
                        .toArray(new StackTraceElement[]{}));

                Notification notification = buildNotifcationFrom(t);
                result.setError(notification);
            }
        } finally {
            MunitCore.reset(event.getMuleContext());
            runAfter(result, event);
        }

        long end = System.currentTimeMillis();
        result.setTime(new Float(end - start) / 1000);
        return result;

    }


    private Notification buildNotifcationFrom(Throwable t) {
        return new Notification(t.getMessage(), stack2string(t));
    }

    private void runBefore(MuleEvent event) throws MuleException {
        logger.debug("Running before test scopes...");
        run(event, before);
    }

    private void runAfter(TestResult result, MuleEvent event) {
        logger.debug("Running after test scopes...");
        try {
            run(event, after);
        } catch (MuleException e) {
            result.setError(buildNotifcationFrom(e));
        } catch (AssertionError e) {
            result.setFailure(buildNotifcationFrom(e));
        }
    }

    private void run(MuleEvent event, List<MunitFlow> flows)
            throws MuleException {
        if (flows != null) {
            for (MunitFlow flow : flows) {
                outputHandler.printDescription(flow.getName(), flow.getDescription());
                flow.process(event);
            }
        }
    }

    private void showDescription() {
        outputHandler.printDescription(test.getName(), test.getDescription());
    }

    protected MuleEvent muleEvent(FlowConstruct flowConstruct) {
        return MuleEventBuilderWrapper.muleEvent(muleContext, flowConstruct);
    }

}
