/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.munit.runner.domain;

import org.apache.commons.lang3.StringUtils;
import org.mule.DefaultMuleContext;
import org.mule.api.MuleContext;
import org.mule.api.config.ConfigurationBuilder;
import org.mule.api.context.MuleContextBuilder;
import org.mule.config.spring.SpringXmlDomainConfigurationBuilder;
import org.mule.context.DefaultMuleContextBuilder;
import org.mule.context.DefaultMuleContextFactory;
import org.mule.util.ClassUtils;

import java.net.URI;
import java.net.URL;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.List;

public class DomainContextBuilder {
    private String domainConfig;
    private String domainName;

    private boolean disableMuleContextStart = false;

    private MuleContextBuilder muleContextBuilder = new DefaultMuleContextBuilder() {
        @Override
        protected DefaultMuleContext createDefaultMuleContext() {
            DefaultMuleContext muleContext = super.createDefaultMuleContext();
//            muleContext.setArtifactType(DOMAIN);
            return muleContext;
        }
    };

    public DomainContextBuilder setDomainConfig(String domainConfig) {
        this.domainConfig = domainConfig;
        return this;
    }

    public DomainContextBuilder setDomainName(String domainName) {
        this.domainName = domainName;
        return this;
    }

    public DomainContextBuilder disableMuleContextStart() {
        this.disableMuleContextStart = true;
        return this;
    }

    public MuleContext build() throws Exception {
        List<ConfigurationBuilder> builders = new ArrayList<ConfigurationBuilder>(3);
        String domainConfigResource = getDomainConfigIfFoundInResources(ClassUtils.getResources(domainConfig, this.getClass()));
        ConfigurationBuilder cfgBuilder = getDomainBuilder(domainConfigResource);
        builders.add(cfgBuilder);

        DefaultMuleContextFactory muleContextFactory = new DefaultMuleContextFactory();
        MuleContext domainContext = muleContextFactory.createMuleContext(builders, muleContextBuilder);

        if (!disableMuleContextStart) {
            domainContext.start();
        }
        return domainContext;
    }

    public String getDomainConfigIfFoundInResources(Enumeration<URL> urls) {
        if (StringUtils.isNotBlank(domainConfig) && StringUtils.isNotBlank(domainName)) {
            while (urls.hasMoreElements()) {
                URL url = urls.nextElement();
                if (isDomainJar(url) || isDomainDirectory(url)) {
                    return url.toString();
                }
            }
        }
        return domainConfig;
    }

    private boolean isDomainDirectory(URL url) {
        URI domainDir = URI.create(domainName + "/target/classes/" + domainConfig);
        return "file".equals(url.getProtocol()) && url.getPath().endsWith(domainDir.getPath());
    }

    private boolean isDomainJar(URL url) {
        URI domainFolder = URI.create("/" + domainName + "/");
        return "jar".equals(url.getProtocol()) && url.getPath().contains(domainFolder.getPath());
    }

    public String getDomainConfig() {
        return domainConfig;
    }

    public boolean isDisableMuleContextStart() {
        return disableMuleContextStart;
    }

    protected ConfigurationBuilder getDomainBuilder(String configResource) throws Exception {
        return new SpringXmlDomainConfigurationBuilder(configResource);
    }

}
