/*
 * Copyright (c) 2015 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.runner;


import org.apache.commons.lang.StringUtils;
import org.mule.api.MuleContext;
import org.mule.munit.assertion.processors.MunitAfterTest;
import org.mule.munit.assertion.processors.MunitBeforeTest;
import org.mule.munit.assertion.processors.MunitFlow;
import org.mule.munit.assertion.processors.MunitTestFlow;
import org.mule.munit.common.util.ProxyExtractor;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;


/**
 * <p>Abstract class to create a Munit suite and its tests</p>
 * <p/>
 * <p>T is the Type of the Suite</p>
 * <p>E is the Type of the Tests</p>
 *
 * @author Mulesoft Inc.
 * @since 3.3.2
 */
public abstract class SuiteBuilder<T, E> {

    protected MuleContext muleContext;

    /**
     * <p>The list of tests to be added in the suit</p>
     */
    protected List<E> tests = new ArrayList<E>();

    /**
     * <p>Create the suite based on the tests created</p>
     *
     * @param name The Suite Name.
     * @return The Suite object
     */
    protected abstract T createSuite(String name);

    /**
     * <p>Create a test</p>
     *
     * @param beforeTest Munit flows to be run before the test
     * @param test       Munit Flow that represents the test
     * @param afterTest  Munit flows to be run after the test
     * @return The Test Object
     */
    protected abstract E test(List<MunitFlow> beforeTest, MunitTestFlow test, List<MunitFlow> afterTest);

    /**
     * @param muleContext Used to create the tests and pre/post proccessors.
     */
    protected SuiteBuilder(MuleContext muleContext) {
        this.muleContext = muleContext;
    }

    /**
     * <p>Builds the Suite with a particular suite name, based on the mule context</p>
     *
     * @param suiteName The desired suite name
     * @return The Suite Object
     */
    public T build(String suiteName) {
        return build(suiteName, null);
    }

    /**
     * <p>Builds the Suite with a particular suite name, based on the mule context</p>
     *
     * @param suiteName The desired suite name
     * @return The Suite Object
     */
    public T build(String suiteName, List<String> testNameList) {
        List<MunitFlow> before = lookupFlows(MunitBeforeTest.class);
        List<MunitFlow> after = lookupFlows(MunitAfterTest.class);
        Collection<MunitTestFlow> flowConstructs = lookupTests();

        for (MunitTestFlow flowConstruct : flowConstructs) {
            if (shouldRunTest(flowConstruct, testNameList)) {
                tests.add(test(before, flowConstruct, after));
            }
        }

        return createSuite(suiteName);
    }

    private Boolean shouldRunTest(MunitTestFlow munitTestFlow, List<String> testNameList) {
        if (null == testNameList || testNameList.isEmpty()) {
            return true;
        }

        for (String testToRunName : testNameList) {
            if (StringUtils.isNotBlank(testToRunName)) {
                if (munitTestFlow.getName().matches(testToRunName)) {
                    return true;
                }
            }
        }
        return false;
    }

    private Collection<MunitTestFlow> lookupTests(){
        List<MunitTestFlow> tests = new ArrayList<MunitTestFlow>();
        for(Object testFlow : lookupFlows(MunitTestFlow.class)){
            tests.add((MunitTestFlow) ProxyExtractor.extractIfProxy(testFlow));
        }
        return tests;
    }

    private List<MunitFlow> lookupFlows(Class munitClass) {
        return new ArrayList<MunitFlow>(muleContext.getRegistry().lookupObjects(munitClass));
    }

}
