/*
 * Copyright (c) 2015 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.runner.processors;

import static junit.framework.Assert.fail;

import java.util.Collections;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import org.apache.commons.lang3.StringUtils;
import org.junit.ComparisonFailure;
import org.mule.munit.common.exception.MunitError;
import org.mule.munit.common.util.MunitExpressionWrapper;
import org.mule.runtime.api.component.ComponentIdentifier;
import org.mule.runtime.api.message.Error;
import org.mule.runtime.api.message.ErrorType;
import org.mule.runtime.core.api.Event;
import org.mule.runtime.core.api.MuleContext;
import org.mule.runtime.core.api.expression.ExpressionRuntimeException;
import org.mule.runtime.core.exception.MessagingException;
import org.mule.runtime.core.exception.SingleErrorTypeMatcher;

/**
 * <p>
 * The Test Flow
 * </p>
 *
 * @author Mulesoft Inc.
 * @since 1.0.0
 */
public class MunitTestFlow extends MunitFlow {

  public static final String NO_TAG_TOKEN = "NO_TAG";
  private static final String TAG_SEPARATOR = ",";

  /**
   * <p>
   * Determines if the test has to be ignored
   * </p>
   */
  private boolean ignore;

  /**
   * <p>
   * The error type id that is expected
   * </p>
   */
  private String expectedErrorType;

  /**
   * <p>
   * The name of the exception that is expected
   * </p>
   */
  private String expectedException;

  /**
   * <p>
   * The tags that the test has
   * </p>
   */
  private String tags;

  private final MunitExpressionWrapper expressionWrapper;

  public MunitTestFlow(String name, MuleContext muleContext) {
    super(name, muleContext);

    expressionWrapper = new MunitExpressionWrapper(muleContext.getExpressionManager());
  }

  public boolean isIgnore() {
    return ignore;
  }

  public void setIgnore(boolean ignore) {
    this.ignore = ignore;
  }

  public String getExpectedErrorType() {
    return expectedErrorType;
  }

  public void setExpectedErrorType(String expectedErrorType) {
    this.expectedErrorType = expectedErrorType;
  }

  public String getExpectedException() {
    return expectedException;
  }

  public void setExpectedException(String expectedException) {
    this.expectedException = expectedException;
  }

  public Set<String> getTags() {
    if (StringUtils.isBlank(tags)) {
      return Collections.emptySet();
    }
    Set<String> tagSet = Stream.of(tags.split(TAG_SEPARATOR)).collect(Collectors.toSet());
    if (tagSet.stream().anyMatch(tag -> tag.trim().equalsIgnoreCase(NO_TAG_TOKEN))) {
      throw new IllegalArgumentException("The tag '" + NO_TAG_TOKEN + "' is invalid since it's a keyword.");
    }
    return tagSet;
  }

  public void setTags(String tags) {
    this.tags = tags;
  }

  public Event run(Event event) throws Throwable {
    try {
      Event resultingEvent = process(event);

      if (isExpectingFailure()) {
        StringBuilder builder = new StringBuilder();
        builder.append("The test: ").append(getName()).append(" was expecting a failure - ")
            .append("Error ID: ").append(expectedErrorType)
            .append(" - Exception: ").append(expectedException)
            .append(" but it didn't fail");

        fail(builder.toString());
      }
      return resultingEvent;

    } catch (MessagingException messagingException) {
      if (isExpectingFailure()) {
        Optional<ComparisonFailure> comparisonFailure = validateExpected(messagingException);
        if (comparisonFailure.isPresent()) {
          throw new AssertionError(comparisonFailure.get().getMessage(), messagingException);
        } else {
          return messagingException.getEvent();
        }
      } else {
        Throwable cause = messagingException.getRootCause();
        throw (cause != null && cause instanceof AssertionError) ? cause : messagingException;
      }
    }
  }

  /**
   * Validates if this Test is expecting to fail
   *
   * @return false, if no errorId nor exception are being expected
   */
  protected boolean isExpectingFailure() {
    return StringUtils.isNotBlank(expectedErrorType) || StringUtils.isNotBlank(expectedException);
  }

  /**
   * Validates if the MUnit Test Flow is expecting this specific exception
   *
   * @param exception the MessagingException to be validated if expected
   * @return false, is no failure was expected of if the exception does not matches the failure cause expected
   */
  protected Optional<ComparisonFailure> validateExpected(MessagingException exception) throws MunitError {
    if (!isExpectingFailure()) {
      return Optional.empty();
    }

    ComparisonFailure failure = null;
    if (StringUtils.isNotBlank(expectedErrorType) && !isErrorIdExpected(exception)) {
      String actualErrorType = exception.getEvent().getError().get().getErrorType().toString();
      failure = new ComparisonFailure("The error ID thrown does not match the expected one", expectedErrorType, actualErrorType);
    } else if (StringUtils.isNotBlank(expectedException) && !isExceptionExpected(exception)) {
      String actualException = exception.getRootCause().getClass().getName();
      failure = new ComparisonFailure("The exception thrown does not match the expected one", expectedException, actualException);
    }
    return Optional.ofNullable(failure);
  }

  protected boolean isErrorIdExpected(MessagingException exception) throws MunitError {
    Object errorId = expressionWrapper.evaluateIfExpression(exception.getEvent(), expectedErrorType).getValue();
    if (errorId instanceof String) {
      try {
        ComponentIdentifier errorTypeComponentIdentifier =
            ComponentIdentifier.buildFromStringRepresentation(((String) errorId).toUpperCase());
        Optional<ErrorType> errorType = getMuleContext().getErrorTypeRepository().getErrorType(errorTypeComponentIdentifier);;
        SingleErrorTypeMatcher errorTypeMatcher = new SingleErrorTypeMatcher(errorType
            .orElseThrow(() -> new MunitError("The expected error type " + errorId + " was not found")));

        Error eventError = exception.getEvent().getError().orElseThrow(() -> new MunitError("The event has no error"));
        return errorTypeMatcher.match(eventError.getErrorType());
      } catch (IllegalStateException e) {
        throw new MunitError("Expect error id " + errorId + " was never registered it can not be thrown", e);
      }
    } else {
      throw new MunitError("Expect error id expression error. The expression should return a valid string");
    }
  }

  protected boolean isExceptionExpected(MessagingException exception) throws MunitError {
    if (expressionWrapper.isExpressionValid(expectedException)) {
      return evaluateExpectException(exception.getEvent());
    } else {
      return exceptionMatches(exception);
    }
  }

  protected Boolean evaluateExpectException(Event event) throws MunitError {
    try {
      Object result = expressionWrapper.evaluate(event, expectedException).getValue();
      if (result instanceof Boolean) {
        return (Boolean) result;
      } else {
        throw new MunitError("Please make sure your expression matching for expectedException returns a boolean value");
      }
    } catch (ExpressionRuntimeException e) {
      throw new MunitError("Expect exception expression error. " + e.getMessage());
    }
  }

  protected Boolean exceptionMatches(MessagingException exception) throws MunitError {
    Class<?> expectExceptionClass;
    ClassLoader appClassLoader = Thread.currentThread().getContextClassLoader();
    try {
      expectExceptionClass = Class.forName(expectedException, true, appClassLoader);
    } catch (ClassNotFoundException e) {
      throw new MunitError("The class " + exception + " could not be found", e);
    }
    return expectExceptionClass.isAssignableFrom(exception.getRootCause().getClass());
  }

}
